# Part of Odoo. See LICENSE file for full copyright and licensing details.

from base64 import b64encode

from odoo import _
from odoo.exceptions import UserError
from odoo.tests import tagged
from odoo.tools import file_open, float_is_zero
from odoo.addons.account.tests.common import AccountTestInvoicingCommon


@tagged('-at_install', 'post_install_l10n', 'post_install')
class AccountTestSIEImport(AccountTestInvoicingCommon):
    @classmethod
    def setUpClass(cls, chart_template_ref=None):
        cls.chart_template = chart_template_ref or 'se'
        super().setUpClass(cls.chart_template)

        cls.company = cls.setup_company_data('SIE test company', chart_template=cls.chart_template)['company']
        cls.env.user.write({
            'company_ids': [cls.company.id],
            'company_id': cls.company.id,
        })

        cls.currency = cls.env.ref('base.SEK')
        cls.company_registry = '555555-5555'

        cls.bad_currency = cls.env['res.currency'].create({'name': 'BAD', 'symbol': 'X'})
        cls.bad_company_registry = '6666666-6666'

        cls.company.update({
            'currency_id': cls.currency.id,
            'company_registry': cls.company_registry,
        })

        with file_open('l10n_se_sie_import/tests/data/Sample.sie') as f:
            cls.sie_filedata = b64encode(f.read().encode())

        with file_open('l10n_se_sie_import/tests/data/SampleEntry.sie') as f:
            cls.sie_entry_filedata = b64encode(f.read().encode())

        # Accounts codes that should be present after import of the Sample.sie file
        cls.reference_accounts = {
            "1010", "1030", "1060", "1070", "1110", "1119", "1130", "1150", "1159", "1210", "1219", "1220", "1229",
            "1240", "1249", "1354", "1380", "1410", "1420", "1440", "1460", "1470", "1480", "1510", "1511", "1515",
            "1590", "1610", "1611", "1613", "1614", "1630", "1640", "1650", "1680", "1684", "1710", "1720", "1730",
            "1740", "1750", "1760", "1790", "1910", "1920", "1930", "1940", "1980", "2081", "2082", "2083", "2085",
            "2086", "2091", "2093", "2098", "2099", "2110", "2111", "2112", "2113", "2114", "2115", "2123", "2124",
            "2125", "2126", "2127", "2128", "2129", "2150", "2151", "2152", "2153", "2160", "2190", "2330", "2340",
            "2359", "2390", "2420", "2440", "2441", "2442", "2510", "2610", "2611", "2614", "2615", "26151", "2620",
            "2621", "2624", "2625", "2630", "2631", "2634", "2635", "2640", "2645", "2650", "2710", "2730", "2731",
            "2750", "2760", "2790", "2792", "2793", "2794", "2810", "2820", "2890", "2910", "2920", "2941", "2960",
            "2970", "2990", "3010", "3020", "3080", "3081", "3090", "3110", "3180", "3181", "3308", "3740", "3911",
            "3912", "3921", "3922", "3981", "3985", "3987", "3989", "3990", "4010", "4051", "4055", "4056", "4057",
            "4058", "4059", "4081", "4090", "4110", "4151", "4155", "4156", "4157", "4158", "4159", "4181", "4190",
            "4910", "4920", "4960", "4970", "4990", "5010", "5020", "5050", "5060", "5070", "5090", "5130", "5132",
            "5140", "5161", "5170", "5191", "5192", "5198", "5199", "5210", "5220", "5290", "5410", "5411", "5412",
            "5420", "5440", "5460", "5500", "5611", "5612", "5613", "5615", "5619", "5710", "5800", "5810", "5820",
            "5831", "5832", "5890", "5910", "5930", "5940", "5950", "5960", "5980", "5990", "6010", "6050", "6060",
            "6071", "6072", "6090", "6110", "6150", "6211", "6212", "6213", "6214", "6250", "6310", "6350", "6400",
            "6410", "6420", "6490", "6520", "6530", "6550", "6560", "6561", "6570", "6580", "6590", "6910", "6970",
            "6980", "6981", "6982", "6991", "6992", "7010", "7019", "7080", "7090", "7210", "7219", "7220", "7229",
            "7280", "7290", "7311", "7312", "7313", "7314", "7315", "7316", "7319", "7321", "7322", "7323", "7324",
            "7331", "7332", "7350", "7370", "7381", "7382", "7383", "7384", "7385", "7389", "7390", "7410", "7420",
            "7460", "7490", "7510", "7511", "7512", "7519", "7531", "7532", "7533", "7570", "7610", "7620", "7631",
            "7632", "7690", "7691", "7693", "7810", "7811", "7813", "7816", "7817", "7820", "7821", "7824", "7830",
            "7831", "7832", "7834", "8310", "8390", "8410", "8422", "8423", "8490", "8811", "8819", "8850", "8860",
            "8890", "8910", "8920", "8999",
        }

    def _get_formatted_balances(self, domain):
        """
        Formats account.move.line search balances for assertSequence comparisons

        :param list domain: Odoo search domain
        :return tuple: tuple containing an ordered serie of (account_code, balance)
        """
        aml_balances_per_account = {
            aml_balance_per_account['account_id'][0]: aml_balance_per_account['balance']
            for aml_balance_per_account in self.env['account.move.line'].read_group(domain, ['account_id', 'balance:sum'], ['account_id'])
        }
        accounts = self.env['account.account'].browse(aml_balances_per_account.keys())
        return [
            (account.code, aml_balances_per_account[account.id])
            for account in accounts
            if not float_is_zero(aml_balances_per_account[account.id], precision_digits=2)
        ]

    #############################
    #       Test methods        #
    #############################

    def test_company_data(self):
        """
        Ensure we cannot import a file on a badly set company
        """
        wizard = self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_filedata,
            'include_embedded_documents': True
        })

        # Test UserError if company registry is incorrect
        self.company.update({'company_registry': self.bad_company_registry, 'currency_id': self.currency.id})
        with self.assertRaises(UserError):
            wizard.action_import_sie()

        # Test UserError if company currency is incorrect
        self.company.update({'company_registry': self.company_registry, 'currency_id': self.bad_currency.id})
        with self.assertRaises(UserError):
            wizard.action_import_sie()

    def test_sie_entry_import(self, wizard=False):
        wizard = wizard or self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_entry_filedata,
            'include_embedded_documents': True
        })
        wizard.action_import_sie()

        # Check balances
        file_balance = (('1940', -15.75), ('6071', 15.75))
        balance = self._get_formatted_balances([('company_id', '=', self.company.id), ('date', '<=', '2015-08-19')])
        self.assertSequenceEqual(file_balance, balance, "Accounts balances changes don't match the file ones")

        # Check journals
        journals = self.env['account.journal'].search([('code', '=', 'SIE')])
        self.assertTrue(journals.ids, "Journal creation failed")

        # Check partners
        file_partners = ('Svenska Statoil AB', 'Hyresvärden AB')
        file_bad_partners = ('SIE undefined imported partner - 7', 'SIE undefined imported partner - 14')
        # These should not be created, they correspond to the file_ids of the named partners
        n_partners = self.env['res.partner'].search_count([('name', 'in', file_partners + file_bad_partners)])
        self.assertEqual(len(file_partners), n_partners, "Partners creation failed")

        # Check documents
        documents = self.env['ir.attachment'].search([
            ('company_id', '=', self.company.id),
            ('res_model', '=', 'account.move'),
            ('description', '=', _('SIE imported file'))
        ])
        self.assertFalse(documents.ids, "No documents should be created by the file import")

    def test_sie_import(self):
        """ Test by importing a sample SIE file and checks that the generated data is consistent """
        wizard = self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_filedata,
            'include_embedded_documents': True
        })

        wizard.action_import_sie()

        # Check accounts and account balances
        accounts = set(self.env['account.account'].search([('company_id', '=', self.company.id)]).mapped('code'))
        self.assertEqual(0, len(self.reference_accounts - accounts))  # All the file accounts have been imported

        endyear_balance = self._get_formatted_balances([
            ('company_id', '=', self.company.id),
            ('date', '<=', '2014-12-31'),
            ('journal_id.code', '=like', 'SIE%'),
        ])

        file_endyear_balance = (
            ("1210", 444050.0), ("1219", -150721.83), ("1240", 175000.00), ("1249", -2916.67), ("1460", 145063.00),
            ("1510", 550231.0), ("1511", 150767.0), ("1515", 16300.0), ("1630", 76563.0), ("1710", 75000.0),
            ("1790", 48007.0), ("1920", 209876.0), ("1940", 604582.78), ("2081", -50000.0), ("2091", -301321.0),
            ("2098", -193179.0), ("2110", -55000.0), ("2129", -55000.0), ("2359", -175000.0), ("2440", -127583.0),
            ("2441", -239232.0), ("2442", -5358.7), ("2510", -261009.0), ("2610", -79255.12), ("2640", 9257.97),
            ("2650", -46626.0), ("2710", -55341.0), ("2731", -70769.4), ("2820", -629.75), ("2890", -70000.0),
            ("2920", -260789.0), ("2941", -80003.0), ("2990", -62000.0), ("3010", -259535.0), ("3020", -230931.0),
            ("3308", -8847.0), ("3740", -2.56), ("3990", -292.5), ("4110", 77284.0), ("4156", 4273.22),
            ("5010", 30000.0), ("5020", 7182.4), ("5050", 1137.6), ("5060", 1627.2), ("5410", 4583.2), ("5460", 303.8),
            ("5500", 5315.2), ("5611", 1506.16), ("5619", 384.0), ("5800", 17386.79), ("5820", 100.0), ("5910", 119.0),
            ("6071", 224.0), ("6110", 37831.2), ("6211", 2021.6), ("6213", 498.0), ("6250", 2991.2), ("6970", 3026.41),
            ("6992", 200.0), ("7010", 85000.0), ("7410", 2500.0), ("7510", 39881.4), ("7690", 950.9), ("7830", 7400.83),
            ("7834", 2916.67),
        )
        self.assertSequenceEqual(file_endyear_balance, endyear_balance, "Balances do not match for the end of the primary year")

        file_startyear_balance = (
            ("1210", 420050.0), ("1219", -143321.0), ("1460", 145063.0), ("1510", 432056.0), ("1515", 16300.0),
            ("1630", 76563.0), ("1710", 75000.0), ("1790", 48007.0), ("1920", 269876.0), ("1940", 387000.0),
            ("2081", -50000.0), ("2091", -301321.0), ("2098", -193179.0), ("2110", -55000.0), ("2129", -55000.0),
            ("2440", -240632.0), ("2510", -261009.0), ("2650", -36897.0), ("2710", -29876.0), ("2731", -30888.0),
            ("2890", -70000.0), ("2920", -260789.0), ("2941", -80003.0), ("2990", -62000.0),
        )
        startyear_balance = self._get_formatted_balances([
            ('company_id', '=', self.company.id),
            ('date', '<=', '2014-01-01'),
            ('journal_id.code', '=like', 'SIE%'),
        ])
        self.assertSequenceEqual(file_startyear_balance, startyear_balance, "Balances do not match for the start of the primary year")

        # Check journals
        journals = self.env['account.journal'].search([('code', '=', 'SIE')])
        self.assertTrue(journals.ids, "Journal creation failed")

        # Check partners
        file_ids = (
            '1002', '1004', '1006', '1007', '1008', '1009', '1010', '1011', '1012', '1013', '1015', '1017', '1019',
            '4545-6633', '7410-2365', '7788-3333', '7841-9999', '7856-8999', '7888-9999', '7899-5558', '7901', '7902',
            '7903', '8798-8989', '8977-9665', '8978-5621', '8978-6235', '8978-8999'
        )
        file_partners = [_('SIE undefined imported partner - %s', name) for name in file_ids]
        file_partners += ['Åkesson & co', 'Decoration A/S', 'Reklamförbundet org', 'Simone & Partners - Sprl']
        file_bad_partners = [
            'SIE undefined imported partner - 1000', 'SIE undefined imported partner - DK122333',
            'SIE undefined imported partner - 1111-2356', 'SIE undefined imported partner - 1022'
        ]
        # These should not be created, they correspond to the file_ids of the named partners

        n_partners = self.env['res.partner'].search_count([('name', 'in', file_partners + file_bad_partners)])
        self.assertEqual(len(file_partners), n_partners, "Partners created on import don't match the file")

        # Check documents
        file_documents = (  # Only those should be imported as the moves for the others are on corrected moves
            {
                'name': 'SIE_1 - eSKD moms 1401 - 140203 133435.xml',
                'content': b'PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iSVNPLTg4NTktMSI/Pg0KPCFET0NUWVBFIGVTS0RVcGxvYWQgUFVCTElDICItLy9Ta2F0dGV2ZXJrZXQsIFN3ZWRlbi8vRFREIFNrYXR0ZXZlcmtldCBlU0tEVXBsb2FkLURURCBWZXJzaW9uIDMuMC8vU1YiICJodHRwczovL3d3dzEuc2thdHRldmVya2V0LnNlL2RlbW9lc2tkL2VTS0RVcGxvYWRfM3AwLmR0ZCI+DQo8ZVNLRFVwbG9hZCBWZXJzaW9uPSIzLjAiPg0KCTxPcmdOcj41NTU1NTUtNTU1NTwvT3JnTnI+DQoJPE1vbXM+DQoJCTxQZXJpb2Q+MjAxNDAxPC9QZXJpb2Q+DQoJCTxGb3JzTW9tc0VqQW5uYW4+MTczNzM4PC9Gb3JzTW9tc0VqQW5uYW4+PFV0dGFnTW9tcz4wPC9VdHRhZ01vbXM+PFVsYWdNYXJnYmVzaz4wPC9VbGFnTWFyZ2Jlc2s+PEh5cmlua29tc3RGcml2PjA8L0h5cmlua29tc3RGcml2PjxJbmtvcFZhcnVBbm5hdEVnPjQyNzM8L0lua29wVmFydUFubmF0RWc+PElua29wVGphbnN0QW5uYXRFZz4wPC9JbmtvcFRqYW5zdEFubmF0RWc+PElua29wVGphbnN0VXRvbUVnPjA8L0lua29wVGphbnN0VXRvbUVnPjxJbmtvcFZhcnVTdmVyaWdlPjA8L0lua29wVmFydVN2ZXJpZ2U+PElua29wVGphbnN0U3ZlcmlnZT4wPC9JbmtvcFRqYW5zdFN2ZXJpZ2U+PEZvcnNWYXJ1QW5uYXRFZz4wPC9Gb3JzVmFydUFubmF0RWc+PEZvcnNWYXJ1VXRvbUVnPjA8L0ZvcnNWYXJ1VXRvbUVnPjxJbmtvcFZhcnVNZWxsYW4zcD4wPC9JbmtvcFZhcnVNZWxsYW4zcD48Rm9yc1ZhcnVNZWxsYW4zcD4wPC9Gb3JzVmFydU1lbGxhbjNwPjxGb3JzVGpTa3NrQW5uYXRFZz44ODQ3PC9Gb3JzVGpTa3NrQW5uYXRFZz48Rm9yc1RqT3ZyVXRvbUVnPjA8L0ZvcnNUak92clV0b21FZz48Rm9yc0tvcGFyZVNrc2tTdmVyaWdlPjA8L0ZvcnNLb3BhcmVTa3NrU3ZlcmlnZT48Rm9yc092cmlndD4wPC9Gb3JzT3ZyaWd0PjxNb21zVXRnSG9nPjQzNDM0PC9Nb21zVXRnSG9nPjxNb21zVXRnTWVkZWw+MDwvTW9tc1V0Z01lZGVsPjxNb21zVXRnTGFnPjA8L01vbXNVdGdMYWc+PE1vbXNJbmtvcFV0Z0hvZz4xMDY4PC9Nb21zSW5rb3BVdGdIb2c+PE1vbXNJbmtvcFV0Z01lZGVsPjA8L01vbXNJbmtvcFV0Z01lZGVsPjxNb21zSW5rb3BVdGdMYWc+MDwvTW9tc0lua29wVXRnTGFnPjxNb21zSW5nQXZkcj4zNDc3MzwvTW9tc0luZ0F2ZHI+PE1vbXNCZXRhbGE+OTcyOTwvTW9tc0JldGFsYT48L01vbXM+DQo8L2VTS0RVcGxvYWQ+DQo='
            },
            {
                'name': 'SIE_Betalfil Bg - 1 - 140102 112969.bgc',
                'content': b'MTEwMDA3MzQ2MjMyMTQwMTAyTEVWRVJBTlTWUlNCRVRBTE5JTkdBUiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICANCjEzRXJ0IGZha3R1cmFuciAgIFblciByZWYgICAgICAgICBOZXR0byAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQoxNDAwMDAwMDAwMDAyMyAgICAgICAgICAgICAgMSAgICAgICAgMDAwMDAwMDk3ODAwMTQwMTIwICAgICAxOjEgICAgICAgICAgICAgICAgIA0KMTQwMDAwMDAwMDAwNzg5NSAgICAgICAgICAgIDExICAgICAgIDAwMDAwMTA0NjAwMDE0MDEyOCAgICAgMToyICAgICAgICAgICAgICAgICANCjE0MDAwMDAwMDAwMDU3MCAgICAgICAgICAgICAxMiAgICAgICAwMDAwMDAwNzA0MDAxNDAxMjggICAgIDE6MyAgICAgICAgICAgICAgICAgDQoyOTAwMDczNDYyMzIwMDAwMDAwMzAwMDAwMTIxNDIwMCAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIA0K'
            },
            {
                'name': 'SIE_Betalfil Bg - 1 - 140102 113283.bgc',
                'content': b'MTEwMDA3MzQ2MjMyMTQwMTAyTEVWRVJBTlTWUlNCRVRBTE5JTkdBUiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICANCjEzRXJ0IGZha3R1cmFuciAgIFblciByZWYgICAgICAgICBOZXR0byAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQoxNDAwMDAwMDAwMDAyMyAgICAgICAgICAgICAgMSAgICAgICAgMDAwMDAwMDQ4OTAwMTQwMTIwICAgICAyOjEgICAgICAgICAgICAgICAgIA0KMTQwMDAwMDAwMDAwNzg5NSAgICAgICAgICAgIDExICAgICAgIDAwMDAwMDUyMzAwMDE0MDEyOCAgICAgMjoyICAgICAgICAgICAgICAgICANCjI5MDAwNzM0NjIzMjAwMDAwMDAyMDAwMDAwNTcxOTAwICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQo='
            },
            {
                'name': 'SIE_EAGWP.jpg',
                'content': b'/9j/4AAQSkZJRgABAQEAeAB4AAD/2wBDAAIBAQIBAQICAgICAgICAwUDAwMDAwYEBAMFBwYHBwcGBwcICQsJCAgKCAcHCg0KCgsMDAwMBwkODw0MDgsMDAz/2wBDAQICAgMDAwYDAwYMCAcIDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAz/wgARCAFMAbgDAREAAhEBAxEB/8QAHQAAAQQDAQEAAAAAAAAAAAAAAAUGBwgCAwQBCf/EABwBAQACAwEBAQAAAAAAAAAAAAABAgMEBQYHCP/aAAwDAQACEAMQAAABv8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAYwD2Y8PYeHshPoHkA9R4B6EvASHtQEg8R6B4EyHsQHkyAAAexAkkAAAAAAAAAU35H1SNdX08jbHmpI2fPSZtedd2Tn8M14L07TjhotClDklmZwxlqgpnObjklsQQ8MU5mMsDYc5uOI8Ok5YZS6DUa4jdWzoWAAAAAAAAAKKcL7ZFOHvYVx4xbli/Ux3B7vyCYtzyqtBLkkKupZLRpNE2RBh1T5BOmFiJaV695lE7DkR0AcZ1CNeq5S3AarR3w4zA9NRhaSsPimTw9AAAAAAADwojw/ubC0fT4W18MmpJez5qStnz0m73lXdk4vbNuYTZquVtxzHLLlW5ysRZKDrR1xKJauZ2ROgyRtE89O1KLanTSyTZ6qsiKtzjkkjCJLKsy1WSBL0APQAAAAAp1y/pSHGcWcGPZ3xnb2LYmfr/ADiQ9nzSzEpNmM1Ua2SVUyxYi2JXCyVqpMU4ps1LQ5quyJ5j06TiV9NqU6a5RZNlgLdTfs4BaPJcNYX42qJ8D7S+GF5ZdBYy8+yXU+c97FkAAAAAHkQFWOX9Wgfk/VM0I1cM4dv5BY3p/O90ymwxmvQnniEqxWiywVelIys0TVDrZrWLARWt60M1WyskeY5jniMIl5TO00IzPZjcaxuGSerDv/OvzP6Dcddp1ZcHVfUvt6L897VMgAAAAAAIL0fbVI4P3jeqkRozZ2vl1mOr8xDA8RsOE1yTxQhknimHqr7WzVkzrPnpRGCU6szzan02mV0QIluzV3JQjQOUQ5cQ4UI4uxLd1+tQjzv3hUrvO7Pqu3N5+6vc+H+y9AAAAAAAZGDs0B83+llCKJkaUx9n5lYjsfK31SzUu4oo8a2YGSFqCrUiC5CJbxNkRuraO7Kt2irlJjes2rZFeIlnNinyJc1YaljtpKRJKHaIskwdMOU60s3B1qJeb+9K87LsvhlHd8PbLsfJQAAAAAAAOWmT5veb/Te3FvcEc+VOx86tJ2fkrrx2al45VXlSWPd3QVRqHaN9EonRW0bXhCk1qRSSvSUc3WXa836G00IjvhVpPMelCfJMHIjjTwncJBiMnX7lM/Ofd1qcznvgnrp/MbEdL56AAAAAAABjChPn/wBAs3U9bpjnSd1/n1xOz8g7q245cCFgRzpg0JdpyHIiQjZS8bWqgWTVGb5v+d/QjSWzZrRdH5w9t/5pzXq85PulonsjtWx6eSJ0CgcZvGVq96lPnvu6zN3DelqOz8Vlzc8sAAAAAAABiiofF+vQdzfp3saUpdb55bjs/IO2LcEzzqqEWR7Rvg0pdiVKtu+t0G2OQKWjfJVKmuzV9jVzi/U27r9Tkr2Z87vwhJ3/AAc92jomyrSUuylet25ax7FntvzuuY2mk1DK0/R0p4H3Nem3ffHenv8A59c+bngAAAAAAAYxFftD3tT+J9w6qaUwdf5zbTsfJNk2whqivRJFlGMq6EuJnzHmrVj99K2bzr+ry21fT7qqgcD7pbPs/I646Pq4V4v1ju7Xlvo71fh/HOFBmumLhDcxNtbeo2GuWQqIY2n6WlHA+4uW5RvpfQD0P5832oAAAAAAABjER3r9yhvnf0aq11pa63zm0XX+SJc51Jj8mvggTVfres9809UrEFPbx9m98zMeWVK8KZ7fLob0PTd7Zje0s7R67Z0Pc5bXGkzfvM2383mfa8X0w5oQpMTlWeaXQc4jCjNmfqempx5/7a6bVeGx568Hd+HZTIAAAAAAAQxlxUy/NjzP6aVqXmHq/N7G9f5K2rbCvGBciFAa806JywZO53RruDW9LAVPpenbw8Grksvs/FmXr9WHdX6WyOf7HDNOnQdDZbu95917PprF9f4HYePGCYFRYRPDNe6CenGLezmjrU9RTzgfZXJa0u73grZdj5NmsAAAAAAAB4eQ+ffnv0Cjavo+bb8rbDsfKph2fM9h6YJQ7V6BDFjDuUz4f19pa/Ykv0Xx9+5NV15ONG3O+jQdwvsfH1dXVl2ceBm8vdE3OMu4uxo9J4n6NZfgqjCvtq2IrZIvDgpZqzXvielaMtX0tNOD9sWoy2S7HyKwHQ8LkkAAAAAAAAAppxfsEVcz6KiZ+DbfsfKXvveN0szyjGtTVItHGJZXbg/VYp4f0/v3/PMP1Hwyz8663bA2+R9MhnkfWU7s87r0+on8fY5rYNO7j302e3vecvxv/nZeitVL1tnjlKvC9W0O5Nxrxhm+mGCdT0tbeH9t7Yy3O73wmStjz2UyAAAAAAAAAVz5/vKz8P7ZzORarufIkaccDU9/YHa+e2N2fIpgxhmDO0PUQ55f7vqvK92/BZdj5rcZxkbF06m8n76kY+xz6uNH1p12ohTqqkbkh9XxUu9z4jlNYXtH0BpKVaF2stectMtv2dvNLx1HeT9P1873udp+hvo/zqpW1skgAAAAAAAAEa6/do75z9FJmXzVsOz8qTseeGc+5NmzpTZbxbPERZ3xEdRMD+d+zo3M95ptqSd6b5Lcnf8AAJEog1/Sw5p+rlnY8WzcPVhjm+5amv6R4ZubYb0HxJx5NDCIi29bORPDJ4Qal9qBM/tH9y/P015f0xya3qV3Jzr/APovz/lEezIAAAAAAAABwUzfNrzP6XlrrfLJI6fzh7tiOrY39U75wRPMONLzpGQi4+hG2l6td3PLcObnp0xKMSlTXfCNZiZIQJz/AG1YeF9bkfPo279N+f3RSW1w+u+u3yYnvRrWK00fi7dttqOLVbuv3atcP7pJ2z4+53c+Oh4B6AAAAAAAAYQ+fXn/AL9D2/4q9+/4rRg7zJycV7ZOexLUpnbFaOm3ZCL8SBHGmhl9VMhc2NiRa4V22OvPM+nMrW7shbvk48jBFnC+rv7Y1bB+i+EKfJ6jH8N7FT9L5mhPsPITFj2rVYuu8MO3Euxwn3l1I0pvRT539A2L6PzCyfT+eB4ey9AAAAAAAAMaqa8n63pnPOG55Ym21TxCSxQPk0J+xdTbEx3secLVIR/fRUYu7JomQWMHp4f5f1Zhc/0c57vi+mPKwfr+jlNrS91PIIHE7uHm8zq6+jFvsOHKmPoKddjYpqmnFp48edzkjF2XBfk91dfZDfaJK9PwVLaxgAAAAAAAYorroe8Z2D1stbfk/Zjar7FcVsYeUnXlrD234pt5MRZ1UbEbUtnD2EbHt+8z6eyeb66cN3ykj6vzyK8vLtJgxRnu+mSOV0E/z2d0dXTXvX4vInyYIthadejro3m+ZxxCDiy8eK6jYt5sNiPceYVdvEAAAAAAABiiONfuwTofSZY2/MZzXbNPKzrXVbco52DixXUe1w4Z2tVr3bYjxO+turV9TX3JzebHnZGh7K02DtSJp+FjG/LtNiwRXvekb+jsJfHyuXo6jx9JlzZADXNuHXxJ3k+VHmjmTGfCClOKXery539pwOnLAAAAAAAAGKOKmep/O+syrm89ndlWNS2CzmtwWl4DtN3W2Hz9E8V272hVrYx67RnB36+27Nb11P8ANwFPD2Iux7Ns9L1ci6Hjo2vy7SYteLN3uO7q+drLt8+71bw7oepZ+7wPZjfEYQb+OZj8h1YL4/R5cu3nNFGNexPc4sn+r5GdgAAAAAAAHkPCvuh9AWpep1VnXa2Ey5snn2L5i6Dgzun1vnpYy6lTtvEwclH9Ftur1lfX9NWTNwnBzfojF19+fkyZq+CjqeTaOmCHdrvP7reZrFvc+79LQzzvZp+/47ptTbMYCLU+fId+FuZuM6LbJh+3x2E7fMkn0fLymQAAAAAAACBKLtT1LZ1e9rmcViYxmzp2vMR1veabFjm1dmdK4YD2KQrmxT3jyw1qd/LB3I/to7OV9LbOt2ZTz8iXdf55Hc8a0NMER7Xo3z1/KVk3ufd2iEeb9Cde/wCMhTY5nTEp5KeOzj8T6Nl87bjPV2NlqzFv6cgdzlSH6PnZJAAAAAAAAAGZh61beZ9NK5PUZo1yWOj4br6PkGvEImLNZ3HDIvWANjFZ/BkpXqe059frbmRkbHEiu/HsrqemnHh42bm4Nn8eGur30xd355Wzo8u7VLVL4n1ebej4iNM/AT4rsJyw5mN4/wBUq4NaEufvb7RYTrctY7ejI3c5vtgexPh7IAAAAAAOCMsa4vVI9dzBIeJjbY8y6tniMytWJTYsnir5KumzjtPgvUl3fcW0oq13y8iMsVrPa/cl7iQhY9SwjCn5+vt7HnK3b2hdisx1r+sdmbjRLl5TNmPSxOlssLy3pXFp4a+8vodNq2Y7nJTNvG9d/T6r06M+Ps28Wd2UgAAAAAA4I2EyNzTN8EifEwFl5pk0mchApnm3DC5MV22aWXwZYabtTYwCVHR9Tw87300X5EDdD5t2YocFav3ZWay68I58NsaS3K9Nw3wthpQjequT5obTF4Hdc/OpXjl9HuvWznc5Nd9fc7NimEZHBbDIG/ovvsaCxnxepJAAAAAHMyaJy6oyalsVsSHM2i5GvH1mGPKoUOyFcc+KwGHai296Czi21l+8j36/y/o0qb3kaYd75Mm0jkrMyWi/dqRTCwEEGuZZvDr1orbervhMeDYZvG7Dh5FoE52+qXrYvscmpvM7q3kx5odXR1OPFLs6OlOnf4/RlpkkAAAAPDWnFbBbWtgnFaE8uqrWxRfYsYs3dWHtVX3PhlXBuRfj6lXOd73bXMnYsDl1vUynveQpl2vmHJVhCbpfQrJSSpYREaVtEBajWyVcrEkSfuLM2+Z0u3g7MMae0o5KTbu6VbtDtr1KebLHJDl39Xjy4ZU6nIlXp6XtgAAAAeHifDxOJinFaEMuBRtWP7R04szerD/qhHPR3YNlAwdmvXM+mKGt2mVl8s58HopX3vIUs7HzHlltiJ4hebNSWZbazF1UPRNp8GSscVlOZWNbMam41/Pb8S4rqmXHLV6wpyOxJHQ0ot1uu79/mvXq6XNizdV9CwnoeNsyQAAAAAB4eJDxHi0H5cPTNY8tKrjzIFIkqqEc9FGmbfrdmmVeo4cctTW7D/5v0SYN/wATAfZ+URMOUs5CxeaktHlZj4jXFexvOyU+teb1Hhr7Oy0xp5jqx1Fu29JBrMUcXsO7p6rSpupWbGo5sS7nwzFuciWvRcjrzVAAAAAAAA8PSEcuPjlGo7dXdT7Y5MmsKZMabTM9tXt0cx+n4t7xLC1tyUef7+WNry0ib/jPnaL8PqFLfmpLpvrLAGFKesc1/mJWFLjbyhsUiTxnoUzDdVy4+vJWC9DsuRLv6vIj2m+4GrI/V57znUkD0XN6clQAAAAAAAACCslNMxHtoUeL3ED5x7Du8ttKGHFw72WUfsvmaRcz7p15eZXLt/FVLNrzbTFXfNzG2eRFnLRc7LSXDfWWGIR0Us2x4Chz88k5aVk+aex347Lmxg1ZYj7ezJNM/exIc5n9n50z9PkKWzhf3c0MpAAAAAAAAAEBXq2rw2YjdyuwpfKPerHzro9eLBwd2z4+++Ko1zPvqjVFPQ+ba9rlS1ueUiTNxIbBF6ieMtJrs60x5U3pPfDZoykM06uR0VmAfnHrejBDg38Gza1kz0Guqc7O7c+t2Xwp1bomZJnc53XmplIAAAAAAAAAjZWK8scMWd2vmx8L67LxHT68FNXaxPH6l5uoPN+sbK7OvY8tnOeQOh4mftvznz3HyX5I5yVet2+rRBMNVLN4cI4cVlOqEPFeh16mRa2Malv6sqeu4qdhyImpscGFlmh4b2t03gPZAAAAAAAAAHEjhTqmvbS2vDmw18uUVLunaxIeHp+y4smr5bCoTjztTWYCsb5jsMonCXPDYYmRsChPwZ/KNss749ueumzKr09MTotGwAAAAAAAAAA1npigW8qFvYjw9tGtYlgplMe1tjMAGZuMgPDw8AD09qwiQ9QHlwegB6BkAAAAAAAAAAAAeAB6BieoEC3p4AAB6AAAAB4engHoAAAAAAH/xAA5EAAABgEDAgMGBQMEAgMAAAABAgMEBQYHABESExQIFSEWIiMkMDMQJTE1QBcyNCY2N0EgJ0JQYP/aAAgBAQABBQL/APYb63/g763/APpM4WqQQt8be5yONFZ6mGeovxCRq4RGRIWc13BNnEqVucLFuXzY2hsGvakO384HSliBNIbABVF50GxG9hI5WfWZKOX8x0lNAsPmehnyAsSbBQVJoEteeF7lGeKvpSbKmU08UjglgIodSdKmUbKmGhngAfa1ANe16O5rGUuvbBADBbUjCayAUh7QQg+1YbM7IR69+vms2+RUJTQyPwl3noU5XAk6rpPBqRkKk0/buO+kh983rXDFAwPSfljgvJzYS8I1m1WGYzRJrJPUFyuG8e9Is50mH+oEQ5rOi7ENsE4wS990ns3cgATbcv5i4Jsg6JzA7bprSBg7KZRE71ZtsR0lxYSW+6hPylwmpuBDC1YgIzX18v8ArkYN9u3DSAAovCYcm54SeHszdCkVNWlMCG/Lzl95D/KW4jXTlLyX9Gjs4ApPBwjmhOk8yzIpOLxjd75pQkEEm5jSSJXKxfzxBPuXbmLBQgx6gukIzpOHMV1klYtdVyjEiVy7Zg2bvERXQlHxY5GQmyDGTUuTupFToRq04VaDkp3pMUG6hoTgTokqrAqMbFxCE1v9fKlAnZC6ReEbPJ6nKVB0F/H5aaV1D+uMw407zhYSL4ssj67RnkRykJGEACRnA4QSKTZGLL0nMQlueG4IvYPiDVsfzBywZv3mN3h4ltzOjp8cFUo58C80i67ddSRA6fceqbrprqSAKa6nvEdlRWcyRHLdCQI2Vm7AitGOLR+XSVqBYz21tFGZrcPlb+5GeRaN7bJN/MxXYMo5uuVdq3TbMb/PNE4/LljT02zVNpFJnt0iWOci7Y/U8QbLpyHPcEVeJpNf4mBZFNmcH7IB82RT0oZqAJmbHAJjpJmXTHS75npG3NSmSlyu28bQvL4umD845NsEK1Izpx26vmQlBREqSCgZMz5HU56x8UDwz3HmR4fIDXppBpRBNoJPdKybEeunFcaoojV2OnNaZkbmrjPoJV9mZHyRqGmLVDtVjglHMIdN07szdCHrBUhQPHDuKO+zdAXj1EnBL6me4/uagBvdbhyUli7GwS8M0talmcAck4/UA9ifkBOwP1SGlJQhFbU9b6WtD5ZGVlnT1og1JJw3Lant3USxsKobk7UEYyWZGJJPLUnBtLXm2wWl89AhUEG6jheOp81hOcj1YpyVWPbG1Y43ii3Q6S66fJJSRdAZR+uJ+BToD3JFXaj1ik02BF2AFh2xRFzk43Rx26Hi8jP1S/ShtO/vH/X1Mhxvm1MSPuRn6qSpfiYPHhbIlupXihvqxApIJ1gysIQRMJbixUn20K7UaqS/vsoZz5KpYW/cpxDcGNxcBxM8+1bXJELLn1q1emv9ebx6x1uprwuJwwyPikl4tJngqKBjjuNdDGupZyR5FtUulIu1A6HUeNCyxHCqqZU+17dUwSjJVeLjFObU6Tp+kxQVMOWuQY3kA+cjS6J/bhhp3Ny+q5S6zd+18vkmI/Glg2WwmcS31aOWNHm1JxiyqqTBVuv6Dp/FL9ogwUbT8qGzNAqjFKHdLgwkybWJYODl36ktwiSyHV6zS4eHwLo1yngWVxpI0uClVFHtGkL/AGKmLxKsRYpSTsEzQJ1eVrHW4yC3IyboN42SEHLNBQ/bJAKpXifUrUSG7Z2KrF52oOdZWRBvjt+HzkYHr/8AHArP5f6v6hlSO8svzMfjTgfEweXfIqxdo0R9Fw+WXT2OBg3kSc4ZcN5/zXvZJcgpOoXtIiXXkGjhXqgdd4pxCxh1ZxisQkiq6KgylJtexunEYD9owhUWhPDtXuvJkc9vYafHBXYGQKBDrKE45ZmeOOfD++NH0Zi4S7IkuRMyj4qkdGP0St38gfqxL9wdrmV4JscPQ+cjP7lTbJ4XY9pRvreIKO6Fibfcnw9cHhtkBRxuwMcNlXwA2cvSmW7nhpxJFCNVelGe6acDOv5KPMZo1VSXO8DtqI9dPmAe8nL7mmmdf68jmSy+SVikU1W6WOeZJx02ZPdHDrwril13Ec4V/IwP5vItweH8kYilYqfHzEDhuXmoWjjbJ8qJYeOTOkyZcPLmYlj2TdGXSZpLzWaGpG+OXf8AnRZdSJuLaoR/ldZ+t4g47rVlAdj2EPhYKR6lzOUUSpGVOmZoJ1EmZyqCDgSu250lMi5BHG7J7Nx+dbkHhhqsUqgkQzmyZcUhLVVrm1Weg4SbnmVQNJoO0YaFudvWuE3i2ljTKznKuSUXbV13IpwCThWaiEDMIp6chtKgJtA4lE0RXlVUPDyeQLixJeXWBFxKra/NuKTiUXO37lqDJToOM2uSKY5eB+YxxdioNhkZlIvBP62VI7zShoDvqcLyj8ILGRtzo7lymeUWhGx3j7RXT8AF3IalV3JnzsrJpE0xRCjzcHkGGuamUreWEIMkLJs0inLeTrF5cVdSMkvaWPyldQd6otRa1mOlsvWK0zUjkmemWwzYHPGW15XnjfI86QaNlYLGosXgh8yLZsWSUbeHwXZaGVlIkKLZ9HE7OUHUamt3FhIVJpW1AVjc3f8AHLj1k2ocUsas/Mci/XftgeMzICzcyPvxGC1CpWx9NrJOpt0afaHsLopGE8mqn52z1PSrJ2tZ6/7U1i31aTt1dxLhRTHT69U9G4O62Qz8ZCEk44I9F26sb4AhnS9RhIst1srnIMvI81hFmQujNwMqskCpGu8HIgIl1jrI3ni6bkvRZLrlR8Py6hKoV44OD5yqqiV052jV+Lt6UXwsGqjVhm//AI5OXlJpe6jgZj3Nn+uOsjx3lV+N78Q0uC9QmsT2RxaINdEhG3apa7VLSjdMqckgUUItyIN2zogvbFYWdXj7xfHt5WiZ32dm7dl99f34xcrR2zZs0sBMte9FAUTGZE6qaPxU+HxCDsFgS5NeXvO5A0Ssm65EjvVpglPo19l/jyX6E/sargEw0ETyS4bPs4k/9cEL+ZGHYnh+Y9OvfwPEDHdrbWo82CiZzTuFdy132wWXuhch9XJMS/XeOFvtOg5psy7IQqpFS5hF2e5mDVfikJWcvWPLHCZBQz8wmcZ09q9gqtllqsDUn6ljRI3cFAwn0kcAOv0lHSR/cZsfN3RJVubTH0Z+GJwdxDMf8aT9AT+3Y5B3XLPiW9vrdY1OAyOTLX51UUg5PnY9NLEcf5dj/wDgeImO61fjR3SR/wBwYa9IC9Wh3EWGrzKrSyUGxFUtq32V/tTnTborSdaOvkhnFlrvLqAUwpKBkeUC0dBbGuXmVOjlkrXW2yzF/FqwEiooURWURKmL9voy5CJgIjps6Lwxq5bBa7rY4UazJ1Sdv9CwGyf29Zl9iU/RP7fRBd/SpsK1c69Lkn2TuVM5l24fNSQCoSHZhHxf8DLMZ5rQYhTlp3JIxtgwi9I8rFyqTmUvhmqM1ecbuWtcdLXKL6a9ldWpWUr7Qkw9iqy2TgmsM6NYotKBndLKlRLju4upB2wDg2nSAcMsURWwyMRjCxWBw0xFC1pSyYMaHaWSmOq5GvCAmf2bQmWsA1iWdErDuFZRZmhId/4f6vIY6et7bHtmytsi5AE3yXSfmKBsTP3BAxnVXmPoJ+KCT9sHvVVj5vdwDYP4Eg2B4xQSGPkcZ3qJrM0aJf8Am83ckkq3ShQYu6HVo5vEyxgapS3iIhYFetyTDIJ4asKMXcfGFeuMlUD28goTBQGNIsKjiwW+Qa5e9MmsnGSDWJfOwl5tdFjW5Ua9RplB0iSJNBQbHxCyAuIt2O6kQfi5jn/bqxcGlBmZZJbrO4pTrM7fm6MrD8mQ47JUXSlnFT1NkM5ZMoNrUnVTghnI/J8Mxhqaj9vCEd319/gj66yRH+UZEtcYpLWPrJz9Wia03jIicj1nUgiwhXSGY1WkBjmc6pG3hzdqx9bLZ2yWoewox0Ce4N9RtyAhchXFzPWqEL5Zf8oSDFdnEUeRWjhqMNBM7Nkx7NWU+UHyb6DsBpOw5vlUl7E5/uYG4lerdaUuFoTJEdpu3cTkjXKwWTTfzHh06sNfpCvtn9lUlVtrTIylglm0C1UbZ2N5HQUDfKeHiO4x38LxDRva2vEkMgpND+pdFNto5ANqWhUZmPbeG+MTftI5Jg0MlsQLO4Nr2mc6LZ3HUt9DYXdzTK+hj9vUkhrCuVcyTlarVKkW9yipc3+vnH7jUx6lwvMMvkFrb669pponHM0/j40AdyN2T3I5NsJx4Jz+CYSekIaEbV+KANtcvRUpVg8vQS1L1ePsMf7AxCbeIUCqsfap5sW0PB17QvNRa5nDD63iLjevWcPfYNoNBodD+A6/6na1JwhCFe66LoTcnIAB3BtJi73yUwGVxj4dH5hJKrAfIbkd5Gn/AO9MQm5SfirH4lG/Y4Q3zdsH5hyb4ro/wvw31+gctKG91aYXbqq2JQCubItothcG0hNONMHbuRcw5OnGfWyxG+aY/wAO+sYI6DQaEdc9RLTvnDxRBlpGRRdaRREytrhDV2Q6oa6W+hOVsmocY2nTseDfF/h1d9Kfl1jGzWaYXRstOVAbnh82834rP1on7HC7oStkfA+Xcvyiu7fbaSWKuXbW2ttcfWQSFRjcWvVaCry09WK2TamKukRXjqpMuygW6fRQ+s/bFeMsTIGYpf8AYa5aE2hNqtfdmy9RtWkBbJ2p6MW6yFEBIwL1usomid6gQhhfL5iL5ZTriAI1nAg7W10262XFSKNbXUvdt+HR/PfFb/ZQv2NBz1rNi8R9n/EeO2R4IfyxNQSNBeLKERdLmAFVhA5lQ06cL8mSQSFOUL0HEkiZ5Hx6It2Meh3bxRuDaFD646ZR3lVu5a6murox9CfVaU+NNynSb16X7vWT3hSxqyXfQ9siFDmGKW5UWMF1L5ideYZbyZJA1pOAERPY3LbqZGlIQy0xVD8Lbhsd57xW/wCPQR/IU1yp3jFnvQXiTHbIsB+1kPqVg2cQ5Ti2Qh5c00aPabLsGp14iIN1rfCKR7gjMe37YenTqyqpKLpiSU/gWdr2s319CvoFdGU0KnpV/eUnPegqKI9llY3+jIZXuIbIEOosdCPcousUxvKRknPnPiAzi9FGjYCglOfvnuL1yBXlbVP7VYY/fPFd/iUD9gkW/eSeJx3gvEv6XyvftRj7LO48J/HaL5wkBpNbZeWXHWKIzvX8I43mr422U5aT1QmnvD8a+fwLuXjXwv0aYxr9GlEL9G69v4zQ5AjdQGSkiv5EjxKCo6jrbI9xVbxOP3ne07J8ad22bMl0HOOmwM4uv2JJtf8ALFrbzsNiq4oQ8AxsJpW+ScwslZE5YIecwI9CQd+K/wDwMe/sFwtYV+yYhU51/wATX+9K3+1Wa2nhZnFcmax098j2zriI69BChIGbxFRdnWlLw35tB9BQLupSW/BGK+NkD+A8MJUJGstpRT2Cjy6/p/Gjr+nsbr+nsbqyQSEJcp8NoTHfo+yc1687hZz3FCylGGk4mLYLpPETeVY/wNBpTk54lI5jCQXhxjI6drJ+LPL8wyMeapHMbpg2XXk3XiQIeSZUN85CtoMAJN419xh4mycbZW/2uHYg6RrCXZzF1Z9pOCttpPdZVkHl9KITsrFZ0erGuA4uGBeTiro9NnXXxU7MWbTHRJIpw70C6IoChfqOQ56MhoUddLXT10tZMVOhep2RUNDUh4dGSnxXk7n4fHndY+ywg2UgqyRFeVvCnZYr8NLEBY5XuDq8TuJLe7pMnPsQa5lcVlsaQraJU7zj7L0zUiW/O0jakmXimmYxhifO7+euOPoF9VG3iBhTyMvX3bjyyuB5c9bj29oyHG9QRLtqCQ676wF7amWb5W2ypepGyhOm8hEuo8iCdJi8lHJ5pmgt1Buj2OPF5HU5t5sdoqWJJpfTMG4iTQp66ehT109WMu2S7qZRtVcSqdaRuoApkTBQAnV8lRaMpDwNfbO5TLGzLFmFg7TDZuA1yOFP2L7gy09Jq9MamHVvkCPvP/vgG+sPNhcOKhOIoRGYHzpHUS4cK1xhKGjLse1tZizWuM66cgyKZzUGPN1by9vA5JDt5ow9aOsJOm8qifN6oqDSJcLrPTwcOR6MrHikNVZtJNsxdrNyR1jTbOmrgHSH0ttcdCXQk0JdcdWv0yZkIiSlOwogh5pavXKmCPWs3CSbrRVUeo+cZsekNjmqvlYfw3r4/SLFRdFbliumDSXk/uUPcckwY7LP/vtScxwmn8/jNNJzFi1bgRuiiktlxZVOn1I3Zzcs0TfNZxtxkKHHEXPdFOpJZdJwZslQ8rtRid1SF0hfWxftqwZXtCwW7dadmPNJEay8buo3oxbVerLuZHHc4Zdr9PbW2ttcdcdXMm2Rry77OoYjU5vrKnyy/iBcfLLY62i6qIjL51UM1rTD3PC6tPmLX4Nwd/Gmc9aXlR+PjtuoTJUR6OZD70cXcMII8pHFynWTRV3IgXg8yzz9lIdLi88sbnY2NsJZLHjIjlGaakLfMxJj7IwL9daPukaQr7HkQkEjkGRXJBorfNhAgwjnTYbIWrSAljLgxRiIv2uTatqb1iyBB5F+rtrbV7DbIFzclbVPErpNzJ2NPfMWG5JwxY3edEjKlShizPiTlTuK8ZDs/C5KSR4yOqkiV9FR7gHcrIEFVXHHzeTpDw+XCuyFmrz6AeVXHs7OI4qxFO02KxgAGbtAKJfQ58rG/wBPQqGwxbMq0XmNjIxlkwowVWrrtkQcjZUQ6tIiyKuk7vG7Hx/BHXlb0KiMKzSIqZaY6Z30aeOlkZA7CDTbK26Lr1GjCLVRgDdVmPJt9e/BtdbE7UUb0ZVw1nrMr08wYYk0GCVufJqxlOSURmPE2/SfoTUeZp4bpKJXTi6cXowEVMmYSsha0+/wlNg8zb4mM6rwCiipllKza5CmylTyillfDeKTdJq3S6wpD13eTSF8iqSAnka08KaMzc23ksRhxqhVQWyHfWpntPpj5MrG6Jd0eiEBorkdQJBtOLmr53Inn6xCLpKzGRI4jq6xaxDSC7jvZczfyuPbk6SH18lrkbWqQBVJhRUVnE/NnRLlewVxk8uSVQYpESrDQdS1YYJBlUu+IlSAqzYNiOGclEdnYXbgUbNgvinn64zCk/Z/w8M8oolJYpTAsWQgnQcn6iuQ+S0TRkeMxXqYhKq3iteSmqlESlIeiVxGu2qZHaJia41eJPq03amhqs2cln4VCMCVZpX2LiSzUDCUxZ05RnFFW12gaq8c6rcESDRigGakf4GYjFayTpdi4jqao0RknCpHOX5odruH20dTgb6yj/xCBvk4k3FCThyP56TgnC8zg5io3z/nPHq+PMg/r+Hhdx6szqWIyG7ZkUessIhrIBN4qnLh5tTAEG2UQ21QB3r0F/vOZ/aK4PwJodj1z7UgbafuWJes8hDi60jCFSVcRxnJFzNIYiZXdrdM2ZGKH8DI9TC1sFKG9TRjMevUHFWxwZjcZ+kOPacK462JXnBNPKus7G7QRpjHx8SSSZGOLZHgGGlRWNihwLjFmNDQt+yNUIe5wE74ZFwfYy8OUaylVI9E8NUYJauNQFxyKRwoL6G79Aa8i3NDxTqPVutffTRKk2fx0awiQbSr9LrsmNXeRwyEK9VNDRj1ApIBdzNadskXOlmYoaURdq6ZwSIHR4AT+C6JzE7ENm7MAOVME13CPUWBvrt9GbaWS5sxjQEqMYUoKxIGN5MXrRrAG7p4h3KxY0mhjCgLEvTIigAa4a20Yu+lEt9Ik46cJ8yoF4E2944bkI2ANHbAYUm4F10tlDH1x3HjrbXHfXS20Qf4Il3HjoCbaEvvCX1464646293hrhrpevS0VPYwpa6eunoiewgG3/gIb6DQ+ug9PwHW2tvw21x1x1x1x1x1trb+Vt+G347a21t/J//xAA2EQABAwIEBAQFBAIDAAMAAAABAAIDBBEFBhIhEBMiMSAwMkEUFSNAUSQzQmEHNBZQUkNgcf/aAAgBAwEBPwH/AO4b/wDb5Owenlpuc8Kqy1QSN9KqMkU7/wBrZVWSqqP0G6qMGq4u7UKeTtZGIjur2V1dalqWpalqWpalqWpakXIOV1dalqWpalqWpalqWpalqWpalqWpX+wyZb5c1fFO7WXx1jaydU7bBMLZBZzUMNpb3ss1xtjqy1oRFhurKysrKystK0qysrKyIQHGysrKysrKysrKysrKysg1AfYZIbeianBuqyMQ91LWU9P6yq3OVLGPp7r/AJzLf0rFcSdWy8xwQ3Cur+GytxvwKCsreEo+O6ugUArefgOZ4qSjESmzuwdlh0+I4rcwGwX/AAmonN6mVD/HtO09TkMhUOm91mLDYqSblxoA6fNKCur+G/iur+Cm3laFSZcoZ4tRYpcl0Tt07ItMfS5S5FbpLg5VDNLi1e3mf46qAC+NbjsnNJC1lsaztHpe16sPf7W3leyw5l52hUTtADFKN0BZYhOIqcuUz9Tj5uRZ9Ffo/KHqUym9Cz6OhnCyt4z5N/LsrcMF/wBtiZ6gpuyJ2CzTLow9x87AqnkVrJP7URu0FSqX0LPw6GI8Qj4T4ChwCibdykZZ/G/lBYD/ALjFH3ClTOrZZ5qLUzY/z50Dy14Kwmfm0zXBSp3pWfW/TafAEUEeN+FkE7gLAqloJpuqJqfR6Yw2RtipqF8h+iFPSSwHrFkAfEfFl7erYmdwpE3ZZ8nvM2Pzr2WS6nmYcxSFSelZ9H6dpQPSrq/jsjwATkVBFqeGj3WBYdHT0gAbupcHZUNvI1UOAwRXLQs/yRNLIWDdXBHhuh4W9llkXrWJvdSJm5ss4T68QcPx5/8Ajyq+k+P8J24TvSs+f67Qh6VZW8V0N+y5RtchW/KFvZOQWTsK+Jq9Th0tWZMejw6nAZ6lgeItrKZrx3Wo3WetfxxLlpsUfBSUtO5n1CqiioxHdp8IWVx+uam90/utWm7li03Oq3yH8+fkOr0VnK/Kbu1eyz+bMaPJsonEkAKiy2JqcOc5VeAzxONt1yHBp1bWT2qOIuNgsCpIsKw3mv7rG8TdW1JkPZZMxoQHk+5T6iVw1N7rH4YDA6aoG6lPc8BxFx4h3WVv95qb3TzusRm5NO+QqQ3cT53usuVHKrmOUBuLoepf5AHQ3hDTvlP01Y9irKy7IN1L4Zyc0tWVsKZI7nv9k+JuqzNlG1trOCxPLsVQzV2KxOlNNKYllbA+Z+pn7BZkx81B+Cp+ywXJskw5lRsFh+X2UVWHx9gji1rxsG6qZBKdNY3Yp+UaGoj6B3WNZOqqS7ouytbY8b+KNt1liO1exN78M3TaMOd/f2FK/lvDlhkmqnBX8l/kMdDEAqeodTv1NTnajqHFyidZGYOOyLhpWDYi+m0lm91WV0cDGyy+6gxeColDYisTrGxsc4lV07pZi5YfiVWY/g4fdYBlGOFolm3cpXavpMQibs2y+EYHnT7qeibI2xCp3voqsQu9DlPTtmbZyzjggo6gvj7FFBW4EoHjDsssn9exD1JvdZ+m0xsj/P2AWTanm0LSnjqWP0cU9uYsxUMcD28tXsbL38O7RsqOjlqnCOILAMoMp2iSfcrHqGCopTCFAI6GXWXbhVeLyzuJJUhKyDRtdI57+61aRYqN408wqnN0RYprrhYzHr0kdwoh02KznQCSj1Jw3QTk1FBf2rEK5ssqG9e1e6Z3WeqnVWCP8IfYf48qrxmH8KobuCsSjLrWWbwWyNR1ndGCQM12Wk+Bpsd1kOOFwLvdW2WYZH01OZohdVLTO8yIQOI2RBWQalrXuY7un9YTKd5GlQs0py7dlI0SPv8AhN3Flm0fonKTugigigoGskby/dPhfGd1ve6y1E9lcx690CszVHNxB7l7/YZDqOXV6PyqjdVYWcv3WrDMMbJFHI/0qowqCopdUYWM0Bp2b+6P98LJ/ayyvinwlS0exUT9YDx2VTE2dhjcpssRwagwLEo5ISWDhheJGlqBIsNxGKtiEsSFwE6drPUUK6E9inygbtTHXB3UEwYC56znizfg9DfdP3RRQRQUBOoFqrWxvh0O2Klp3Ml0lUcDRyw0dk1SO0xOcquTXO5yaPsMt1BhrmOUnXGHBYjXRwnTIbLNczHva5husJx6OGhMDxuFTZo/TiL3WYJ3z3v7K2/gY+x2WUMcNRFyX90SLaVNUQwjrKzXLSvN4TvwuOywHMMmHn+lV5/6Po91VZgq6h2rUsPzRJH0yKhzGytcIo1Ae6xqSQUby1VtdNNs89k3gQhxorc4BZgpmRhkpKxCsjme3kjdUcbGwsLjum+yzDPyMPe8I97/AGNNJpeHLDZ+bSNKz+76jdJTnE90ymfp6UaeRm6fUSl25QBtYoK9lc8MNxCSkl5rFW55qZG2YqjFKqb1uVne5Rv7ogK2rstPsje+y7DZZMDucotu6xUXpn3TjubJjSTYI4LIWak9hYbFHZC3Br7HUFNWzTDS8q5abrBa6SSqaHnZMHUFnyo0Uoj/APX2QNlkyp5tC26z4zVI3So8Oe7uoY+WNKrI5S+wXw8p9lJC8e3COjc/svl8y+AmXy+W26+BltZSMLO6jie/0rkFnrUrm+yYwvPSsIypBLTtkk7lDIlGY9axzLkFNBqhCylBI2fqTOyxAXpn3UdEZCbLDcLeJtTk5vssYogRdqLSDYqGjdJ6VBg//tfJ4fdVNG5noTYnfyWHAtmBVMbtaVn2r1zNi/H2f+O6q8bovws3RBsrUDtwITYwFJCHJmGgG6iYGKVxaLp2KvabL5tKvmsqmndIoKoxp8xe5CLpBKw+KHQC1YP/AKosg46bLGhaIqnrmUzwXKPMtMVX5np9BYfdYfFoGoqnddBSNuFLhoe66p4AxOciFy1KyMN7K7B2TMeqB03VZ+rfzHr4GNfAxr4BmlTMDXefkOq5dby/ys4bysQ7cDx7Lun7tUws7ibqxTO65P0AVhMljpKwmsbyNN1TzgtssZb9MlYsfSU3sFVXuFTftBUwtwcnbcAgOFQOlNbcLlC10BYbKxVipDZqmIL/ADisuT8uvY5Zsd9SNW342VliM72KFs0llLBKzupGg9/DGOpPZ+mCw3Z5UtXNHJ0lYTjs4HWq+o51MVin8UPSFWdwqWTpCZXMZsSvmMR906uhvso3hwuFcLUFcLUFO4cvSo9jZHYqGPWVKzT0q/8AFVb+pO7+d7Kmfy5GvWOyCURP/pAX34WVlZYp6gsNLQ0LEHAhPaDdEcCgoG7hTN/TrDv3rBVGF803WHU8kMvX2UsREBKxX+KHpCq+6od2KtNpLLU7Ug5UpIgujWSX7r4uT8r4uT8r4yX8r4qYncqOQ3uUHalSShrlVTXkunP3upz1Eo/YR1HMo4j+Faw4WVlZYrs4XVFibGNsVU4pG9tgmndO4FNVA28gCrumnKwZhdLuoMKdILhS4A9z9QVTT6IC1YwOy/gqvuqEgMWIH6iKadlh0eqmUws4hBWVkPUF/EJnoQB0qylTz0O+xwaTVFy/wg24TYkY1y1y1jN+YE7ZyjIJTzpG3A8GrBmXnCxsaYgsCpnE3Ko26Wd02rYPdVU7XMNljfsh6FVepUNNqZdYiPqIlN3WDs/TLEGaZyECt1umjqRZ0hQn24WVQVN+2j9hg0gbUAE7IUjXbsXy1yOGOXy16+XOWZmaXom6bsu6CdwassU5ml2WaoDFELrK8OuAuKxjG5WSGFqZjNT3usAxB9Q8tkWPt3CG7NlV9gsEjaaXqWKfvFHumjdZahjNP1LMULWVhLV3KurqJtyqkDSoeHsqkqq9A+xhaNSgrpI27FDGZ/yvnM/5Xzmb8r51N+Vi9aZ37o7LumII8AsHq3QOuDZYviDpm2JusLxGWFmkKqe582soU2sDQqKnkpm61V1PO9SaLN2VUVS1zmx2CxF13XVtk3Zqoa17I7BYlKXy3KeN+NP6046mKJNTlP3U0WpoXw1k6BcpEIeZFsta5i5i5i5iqjdDbg1BFBMChdYKV9ymHTsFhOGiZ+p6hyxDs/8ACx6mEFO4hPq7r4tOqroVRCe/UjsmqmksqpAXWlaVTjqUG7Smd03spOyk3Kcy7EYimUV0+hI7KSn/ACpGae/mAoOWpalqWpTLuggmJ6CiTeytumDe6wE3Ynz/AEtIWazamPBu67cOy78I32UkmpBFBQd1Rm4K91GdlJ6U3dyZH0KZ4amyF3pTzKnx/lS01+6c23ljhdaldXUvH2TE/hGsGohVVLYUMhNe7uv+ANA7plE2in5ITjaNZmd+lc0ooK6v4gE7hDsLrDjuU7uo37KaTpVOLuTFURKKEgAhGQXspDfqTpf4qpZ5l+N1dPPgCPCNZelMVY1yizE+1rKDGho1yKoqubVFzQrXYXFZnbalJKKHEocQrouRcm/tlUB+opUxS9lR90TZt0yp1u0odLVWtsSWqnu7pTqX+SqG7IjfzLq/C6cjw9kEUEHLACBVjUpsSdDMbqbGWv0uPZUk7ZanU3smxl8Yss4bU+keS0rUtTEXMTT9ErD3fWCmBuowVPsFQDqUrfZMp7OuEJQGWKdHzNwiwR7qeq9gpzdP7/YO4e6bH03WgosKDShEVlqPXXNa5YrgUuoujCq6ORmxCo8SdTS9lFm0cqyx3HviYyw+WUAv/iVIbShStcomFVLFS9Kh+sr6HWKqCSdlROLY91UE3unusETcJ3f7A8B3VM39OVhVBDJSBxC+WwHaydh0Q9kyhi0nZZdIGMNH9lCmYd1iWD08ouAsaw10VWYgqFxY/RIsT0a+njfyCgnu6VD6wnS2CjmU8paUyTU03VPI6FyJY991UhnsmOYGKpnCedR2Uz9It9ieAUD/AKBWA/6YVrFSXUfpKwO3zlv/AOlREOUpN7LG8MZNX6wpcuyufqYsWoHwv6+F/LPpUPrCmIsoSp7WR/aJCpMS0dMilfbrYnTu90yV4cup3qT3iMbJ79e/2JQCt0pr+iywvF2RwBq+eRlHGIynYywBYXU6MQEip8xta0J+ZGWVRjYE+pf8laAsaxHnnzbqM7o1jCEytjClq2EJ9UzlWV02Qps65yfUEo7/AGdl/SsmOtsuYUJCFzlG+ztSFU6yNW+3dS1RJXOJTneGyt5DVrWtF+yLvbi1XVwr/bDw6lq4NR88f9If+g//xAAyEQACAQMDAgUCBgICAwAAAAAAAQIDBBESITEFEBMgIjJBMEAGFBUjM1FCYRZgJFJx/9oACAECAQE/Af8Auufscmfr5+xu6s4zwfmqikR6iRvURrwkOcRPJgwYNJpNJpNJpNJpNJpNJpNJpHEUTSaTSaTSaTSaTSaTSaTQY+wv96g4pmn+jT/Z6hTmWcsxPnyZMmTJkyZMmRMb7tifbJkyZMiYmZMmTIx/YXks1RSeBTweG58ELCQrAoU9KE9zPkx5X3Q/Oh9sCF9tXtNVTIrH1ElTp8ivYLhH6jI/UZ5LWrrQl5898GOyJd8eRDMGBCMGPJPgqXU0U7uR+eYr/wCpgydQXZPt0+W30MGDJkQ35c+bPdearLYq5IZHggt/rXkcxFz2R04x209kJGO67Ifkz9hce0mQPgtF6vrVlmI/d2R0wfZMkIiSYhi8kRkl2Yn2x5UPzXHtJkD4LBb/AFprJcLExCOm90PtnshjF2ZEZncc0nuaoDmhSyj48zGLyXHsJkB+0sFt9e8X7ghHThfRYuzEMyXdy1UIXWSdSp8FjJyiNbC7rsx+W5foJECXBZr0fX6hHfIhHTuReTHleEyMvUMTF2uKigslGg61TLLmhoka8o6c/Rgz5alSceCNScufNc+wmQQ93gobQ+vfx9GRDOnci758mTJ4bbG9Mu2ntJlWq61TSijSUFg6hS+SDwWMt8GN+2TJkz/Y/wDRkz3wXXsGRKKzUwJfW+C5jmAtiR07kXA2ZNjbvgwVJuKKajKOWKOCVXBTa+S+r6fQi1oKEdbK11hZRXu9VPSyFP07kdvYUpzXvKNfP0kXn8YxFov3Prx4J7oqrDJLY6Zz2wY7YEM1GR5e0i3uFKegqvAo6hRWCdKEnrkV62Y6UKXwz8snHL5MPOBTcWKrLTsWtZMjLPfJgSGu6Lz+PvYLMj5+wvI4kS4LWegtKmvOfKhiJ1FHkub7PBaVWp6hT1IhsR4wX9TEcEG28ItqSz6i8o6FsSm8kNyKwi3rYlgp3MZLT2fZkeB90XXsfZs6csQ+x6hHclwUpYLB5TMme6GI6pKUeBPLLPEpaSnFJDZ/s6qvTkts+5GpRjqZc1daJJJkZYZKrsRlGJ0+in6+zESI8DIrJJYEXPsYhotY4gR+wvo5jkjyTOm8FW4aqaRVW54KceyH2vaOtEvSUXo9RQv3J7i9S2EyrR1ojQ8Eq1tRLD4NBo2GsMkkzpsqie4n2RIj2i/UTS0ix8E5vcZzJIisD+wuFmJ/kRpai0p6SraydXJ+ValqI+bqNrpepHIozzsWLnjcbEVqWsVgU7OmvgqWCl7StaShyVHFclKnGTKEMIZnsxdqfJce0pQcE8kp5bHyUI5rY+ymissVDp+6MYNRnsjHkq0lNFPp8UyNvGIsdlIjyfI8m5fPYmtyns0clSahHULrUNeClVVRakLung1ZGlkuILBLksF+7n7O9jiZ0/ZEpEskGaxTMjZrNZ4iNaIyTJSihMS3G8FW5amfqEteCjdNyL6eUTjuU1wSng6rdJUh/wDsdDu2vRIyvgnPB4542SnM1xJSiyfLOnrbPfP1/k6hHfJY8M+RiGIdQe4hQPBPBFAlAUe1RleXrJS3KL9SLiplYNDkykyFTWzrL4gTfq0llL1RIzxEbyLvc1JRiQvZ4HWI38obH6nVP1SqLqk2y2qynDP176PpyWPDPnvkyMWSMiMu+DA+1VFWnmZOO5QXqRc/yNFs95Fus5LPk67/AIlRcFlHEomPSY8l7B6Bz0niDqo1oyiisss1in9e5jmJZ/J8+a46hGBS6rCU8EKkZCfkk9iL3JEaWSvbEI4mi4/kZa+5lD5LPk6vRdRLSh2FZpbFvaVlNZP6yYMGO1xHMS6Qp/AlkawaiyhuU1svrvctlvIhwZMmTIt4HWXJTOmS1VSOIxRTfZIaJcEXuTeEQrYKszxPWXH8jLX3Mt/ktXyWuGtxRythr1C9TPDPDPDNBOnmLK8Nj/Mp+8qCiWkN0R4+w04nIUvSZMmTJFao4R1DolatLYsPw/XozyyVPEcMgRIkyfBDkqcEquGfmkRqLWV/cWnvZQLVclr7RcDIi7aTSPhlX+RlTTrNcdR4qKL3LT+VfY11gT7ZMmS3IRJJYOX2piJlXgo8leexjUyVOSZFNSRX9xZ+8olo+S19ouBmSnLJsYMFThlSX7jLmK5EItUWa/eQ/sK/BumeKeKeMeKWu5wMZJEGLglyXMsLctJb7Fy3kt6GpErdZLyGGit7yz95b7yZbtqTSLTgXBMqaihnw8Ci9JiZiZVWzJLFQuYjIstYnTN5MXH2HJ4KZ4EP6PAh/R4EP6PAh/RRp6RLPaRIgfA+SrGEvcQpwh7SpSUmR9CPEwyvUUpIrx9ZQliZb/LLLCqblpyfJNFOmmYwJm5uXHDLyOKhW4PkjyUNollcqDkLqGwrwjckZ5Ht9XSYMGDBBd2TKY+D5J7kY4NOEVbjBO7b2KD1TQ7GMhdPij9PgijaQRGGkW5JEEMUka4mtFzL0nVF6osqcEuSC9RTXpIciuKaJ3rXBTv38op3HyihWU1j6ufLAyS7VCBLgXJx2xlMuVhmPWWq/eRTJjZgZwNEYnJODFSiSpIrr9s6ysKLHuipyUVuOWImtlGEZsnTcHuUo03EjNp4RRr6OClLK+xh2faZAlwfJXnp3JX7iQ6i2mOeuJ/mW6/eREmREJDRjs2khuLMGCq84R1qOKaZDDRUpSb2KEJZKzSRUeIYLappkTr65bkaWXsU6TjsKj8ltU+xh2faZAlx2uV6CVunDJChsyEcRG9y3frRAmQGRGZ7PDPCQqZ4ZPaokdb3oFJLBLOSg9y7ccGdUskrXHrGty02e5KSZO5+ClPchx9hDtkZU4FyTXpIrYut0eFmJGj6SpDYn7i2/kEMj2Q+9bL4HTr/AAUoVsFOFZSJybuonVY5oMoSRVqJFvNSex1BrBCXrPzOZ4Jxc6iaHVaeCFVtFOlkprBS4+wh2ZOeCVRYFOOSdRYI1I45Lp+jJSuNiFROJUo6kSs8yKVtpnnumIQ+7HgRPY5uTqUM0WU5xRVnFlrOPyXUoy4KlDfY0ZRbY1FeK1lFrJBD9yKXC+wgZJe0u5Yqo/EnVbi1rtQZL8U3q3TI/i2+lL3FPr15OsvUKWqzTM4gUrhpFvWzEqf6IN+VD7vshkIfusvf4WU92y42LWGtbnh4eIn/ANNM0LUuB5fJQpCWEWlLW8nxj7CBg+C6hmoj8WwXjPJPkpyjrLZJ1kQ2s0P2keClcaY4PzaXJSqKfliMx2YxDKfvZd/xSKHLLs6fulkpL/yEi86brWqmQoTjtIhRS5JUosbiuCnbTqFGl4aF9ghS7Tidf6FK5qZRU/B1wn6T/h1bUslt+E5KqpMVHFuok7cjbsVsz8s2UKWnyxGzUZM9kM07leOY4P0uomV+lVJFr06pFFtYT/MKQkSopk7M/JFKySIx0mDH2bZUTksng7ngEaK5PjBoPDFEx5smfoPc8NMdBEaCI09LyY747Z+2wYNJpMGPs8GDBj/oP//EAFQQAAEDAgIFBQoIDQMCBAcAAAEAAgMEERIhBRMiMUEUMlFhcRAjM0JygZGhscEGJDRAUmKS0RUgMENTY3N0goOisuFkk/CjwkSz0vElNWBlcNPi/9oACAEBAAY/Av8A8T8jgrJ4KeKFrsMbsNyboGLSdV2PdiHrXxmGmq2/YcvjdNU0p7MYXxfSNM5x8Uuwu9BXOamBsck2Pdgsi7kdXhacJNh96+S1PoCk+KVR1XO3femy8lqhG82BsPvXyWp9A+9SP5NU4YsnZD71G001QDLm3IZ+tYn09QB2D71IxsM94jZ2SpY5Y5Wmsl1MeQzdYn3LwM3oTgIJ9g2OS8BN6E5mqmxM35Kwhmy6kLwzZm25arUzYrYtydaKbYdhOSuYZvQmx6mfE4XGSe0Qz7Bsclcwz+heBqDnbcPvXyapz+qPvXgqn7CtqKq/kL5NVfZH3q2pqsjbmLKCq3X5ixclqreSPvXgJ879HDzr5JVn+EfehByepY48XNyHzCq44YmexbcLh2InUyE36E7vMm67ct6tgcMvGasLpJntj5oLyQOxU42vDyXz7FJf9Ifat8qrhnu4+Soct0jf7luf6VWec+paNJF8j/anFrXg8LuU80Mz471AD2b2uaRdaOZTCd0lG7lLtTa4z2b9WRTJWkFj2hwPUpGDfv7lYDuIYfUEbNjPlJmUQ763m9qHXEqnKM9/PPT9mIbub2qn8kqtyjPfG8/yU7KAbuZv3q2APs7de3jBRubTA2P09yhd1n3o2jEl4rWJsh8Wj5w8dS3O6b0KmLQ114zkTkclfV0g73fN+Spe809nYt5+qg7Ux4g63hLNTi5sPDMSX4fMKzyWezuHLf1rUxB8srvEYC9x9CDnU7aGI+NUHP7I96Gp0u5kh55dACPNmhRy1fLXOkMmPVWw34KotweUdlv+4qsfVb18Fkcg4Z+dG4j+2q3s9y0b9YkD7K5jRtDc+6l1b2tdK+Pd2KU1MeiJiO8w8qmcLAdTd3HtuqLFjc5sereHuva3C/HtUZihaHOCEWPvjsgLKU9Ib7k0bhgJ9i4m2YF7ZoTYe+tBaHYzuRP6W7nC980W38196ZKbY4+acW5F7zZ8p2rOOeSc6/8Ay6s1uM3bl52plQ6lka2N4xm17D0qE4X5uOWV879aaWgva+LBwOfpT38lcNSMZyByHnU0rY398kxtGW5U0zaeR4Ywhwy4hNj1UQcYbXv1dipO8zvcJLHr2D0pz6iqkZjcXaszBojUL45sRbbD3wna7fmFRVUOj+VwzhtnCQC2XELv7qLR7O3GfUooNLzaT0rUPZrMDO9RWWr0ToGGmb03tf1LYpaVnpPvWH4m3+V/lT1FU+PlNJOY+bYWsFIzWNDJXXIu7aPpW1/cU8tmY3WDbuTc71qccYp73wXO/LrQxPxGxFypY+VxxiSwe08cu1QR8tZNqMmZWI696Y9uJ9ngkDfZOdhwbi3EN+Vk+r1rqd+kJvi7W3wy7WA4nDm7R3p+isOqMGJwa83IzzF+I61Edk5WT3MwmMOuXtdvKqRGbNaIwMbj1XQLiLWdY4h1Lwpb142o/G5PtsTSZy4WO9zbo4ZsIPW0r5bL9piualzx1vZknhsgdb6wPFY3uaW9XAZKRsBbNLvazdiUdqeF5HOGsGShwRRvzzwyC4UgbJtOacPWVq9XC2To1gTYsFPrHDPbyUbCx5c1oBVLI0zbUrMmt3KeR9PFjx86Z9iVUuhFG1zYnObqrYgbb0zBpWv3X2pMftXywy+VEz/C75BDL/Lt71eXRwPY4j3KKUtwGRgdh6PyujKnpD4/Ye6T9FaUp5xJJrGx1Oz6/cr8mqT/AAlWFJVu4c1XOi6pxd1G6y0TU5Z5tQb+DKrZ61Y6FkNjvstvQ9RG4HeG7lj5FVR4OJaPvTqxsT8Fw1okyvtWutFvc7WUlfJfE0m13OLrb/q+taNkBvrY2A33jp9iYoG4cWIX87ipDNGIhCBEwB2TmjihjgBb1uVhTRW43KfR0lFDVzs3hr9m/WU3lejqfk/HVAXb6U7kTKbXRc+PMOPXbgrclgz4LbpqazkSynjAIz2rJjJG7LzYjGrtjzGS8AE8tgALW5WV9Q24CaeTx5gHcvk8foURDGbk2zprNmjyYzIbSqsdOJd2c29aQqeS07TFTPdl04UGXvZoPdp4B+elaz0lNHQPyrZxvpZmu8xy9/dk8lV1onyXo2ZCwtmFZtCxwH+pYgRo5mf65qJNFC0DpqArtpaUjp1+Sxcjpt36VDWU9Cxp4mpt7lZlJSXfzfjN7pzKmkigDdoSCQOz6LJj+U1E0MsTzHGyzW5dmaoAQ2HvcZwvAZh3ehU1NCY3TTXaxjSO82uQU1QR7sBapjJVM2hrGgsyY1TSupZJ6aMYzMxuw+3QT0qWQVXIaeUkNggYGsjHtv1rwkstQ951hvdjhw86bHGx8kkhwsY0XLitAaZqOSF1XLqpabWjvV98cnRcHzJhYyKNzhis7gmkxRG/Uo9THC1t7HKyYBUvaRtd7ZbiiBUVhO7Pd7E9rIq6QsOG4ta6EcsdXFrQbYzvy7V4Ot5vS770WQU0kuADMzEcOha2opWBlwC7W3TbR0+RIBL+tTd+c3A/FhYLgWeqjCGzZ/nTYjq3blpUmNrPixGR6U39i3u6Oj34XmQ+YfltIQ2uTC4jzZodnck8lVYw43OoQ1o6dpuSZibGYaiW2Nv5t26x6lk6S1+AUlLFgd3oueX81g95VLRzYNVKzvEjRz7bw76yzdPbyVNGC0QRPDXSOGd+hv3p9BVNijnhYHMMbdmRu64C2sZs9lsTLeMFFT1AbBE7ahcBsSXHqd1KE84A7hxQm1ZbLdzdoZ2yTR1pnltTGyZtfSG4tfxlQRTVU7qGDHHyVmMYiS12MWFnYc8lHLo2j0lDSHGXGojIFr7BHENt0pu7IWWkpZqIV+nqUNn0fFisSBvw8LqhoaejZR6UrZuX6Qjvd0ZtYXO6+Z3KgbX1jGzVkjuS45PE8VljvXJ5HOiBG0xxuP4UyRhuwvGaA/Vv428bpTruH++pXRPZqBNhtvKon1ckZY8ktw5W2U0975vSUX01RqmNhDrdKfO6pe9rXW1ZCB1cm8/mx0qqgjLGMJIu7nC5TxUauWUcdy0leMN70Bk6/jBM/ZN7s0vCngt5yfyz2nc5tiqmA74ZXM9fcl8lBrefJQuw/wBKGsluzWAkX37XYvE+2VVyQzGNmDaAda+S0a6WXGzHsDF9XsXiZ/rCpHa46l018OO3jdia6eTWP5OQNvcLhcM5GfnC7xlyeYcppjYOi3lvWz7vQoKdwMj5s2OGerb0lMwg3EtibdYTG22cyOpM8tqY8bhThp+0UaQDfKZpDbhZuXnUgj09pbR8VYRJLSh2sgJAyyy9qAa4aRo5OZPGLO7C3eFQP0XSS02kKJr5cTXta5/p38Mk5un9KUVFWlrC51XOHyYegYd3Hf0qno6Gpo6zkDGx7BBLct60lHT1xooNFz4PkWuytnuBTtfOKh0lQ5okEer5rhw7E1/U5u6+9O8J/tKqucI1uLPK25aKfGdYMWIFmeLJM2Jeb0tUuIWLqe2aqB139ivZt7/pCpZQ2Z8cjrFsG2b9KZK6CoxPGLMAOHrWkCGTN5g/rb1pn7Jvd0lU/TlDB5h/n8vXt4SkSjzhBSeQqb6tE73L+Mf3L/8AhVfWLepaLHQf+xD/APWgOmUf3Jv7Dov4ylj5TFMyZ+KGMO5oZv8AeVUvFssLlrXyMhbLRRuu9+Fty519/TZPeyqow1lVdx1gzFxmsAdfGMbbbrJjQLuvcD/KnxXuIW5dG9VuMOFpGAOtkWgC6fP4rWYvNZSaTqXPdNVXeW/oxwaE984diLSQL5hYi1t3BvDdkqvS2HAGM5MOs3uVpk0DmTN0w9zbavW6wxsAdaxH0reZU9I3E+WOUPkNrYSTmoT+szzsOKO1D/vLTkcUkTHztwa1zy1kLTYF9+pciNbBpKCirLQ1ULnOY4EZtsRcWKjOODm9a4SjV4bx5D1qSLDfWWzuOpWfPEDfiLHcsFPrZRJJcmnAuBZR46epc9t2uc5ouc+1V3e5G3dHvb9dqH7Nvs7hUEh31DnS+k/l6KpH5+Is9B/ygj1tUf7gf+1RnC/ni4w571e09t/OKqTgluTsjDm5UbsEtmg4rN5uys46m3aVEAyUu1mbMGYF002kLDFhxNB33T+Uv0bTQ7ZgxPwPOLeczb0Kdza+ldijPNmbtGyLo2xyOfSQtwyus1oGLqVQ0xUkzowRJHFJnu3IsqWuYad/esYGPVnc0/8AM03WANd0X4qdzhhLmMZboWvkbE6F9WBvOLcPMjBG4CetOrA44fGTaVt2wxjHNJbmN+8qtpmOfgp5nRjFvWytL6NYTyypnMTGtdZwxM53YE2mfLQ1NFo2Q09LWuaWzU4DsRLR03uDvuqudghGEsxm2045IR38f0hZ/W43VdSvbHaWItu9uMDZ6FTCl0XTVFK6V+GY0TnF2eYJac03VaBZJmAWtoJBfPfvR+Lw876PUjsMJxHgh3pvi8FM+MBt7EC3Unb3WDgR0G91W2uNqPj9cI9TW+zuO7MlQwfo4Wj1fl6apt8nnHoKCB+qpcubQ9PW1N2WXvZeIM7K5bEb8ULtgtxsCneB6t6YNTTOc8HeDlxVJUamKQ15OEXEYjsMRuStE0GmqGPbxiOSkrrlmJuLMW+qgY46zFF32O9R4wVRYtziix3z6fcq6Gn0TTxwufhle9hD53DK6nx4KWZwaXRXx4B0ZI8nHKDPinGZLSRbxtwTpsjm21vJT56h4jipZ8T3HsTqx8b2xnYpoeNuHnKa2YDl1WddUkcHfR825VGlzRxv0XM2OJr4rXa7646b8VhZTgZ5kvUD6N8kGlZnCnhdHJbCb/8AL3UEM0/KJo2Bskv6R3EqtAtiOC6JAzGz28ELULsh9VH4mbOYd7m9CpG08F2sqJ288C3fCg4QOAPTKEbU5yJbnIEAKa380KRoh5jsJ74FUzyxuZLCMe/ECLJkkuzrm4rqts8HOP8AvCd2N9ncoqb9POxvrTR0D8vpGPi2PWD+HP3IJpVRZjn/ABEbh1tQwsmYWPvwzWKpuyPFbWSlozPnWVPcHjl96zp3ekfevkzv6f8A1KjZKamm1zixtsNjlc8ehF9dqzT0+WKYB27IHtWmtL6QZqqWomcIpNTtC7nEcOjoT6egq456nV6wx22mtT9HNbKDWNaXSiTxexWnx1DScN+cHdifZr4Y2kuu03LkY7F1NY3Zi9nQjVYdU2YYtWc/FtvVToKnJbGyfFWS+YWY33pvwl07NySkpO+U7JeGWTj19AUstBVTaI0UD3pmEGaf6zr83sUmjaqu/C1G54e+R0QYYS3gHDeniJhlfGMZ4NA4ITuHI3v5kzNprfuKa8aUlxb2vsDf71UU2k2R01dNg1cjPBzYfY7qRGLbxe/evlkJ2ekZ+pC2HBa3PH3Iwxm2Gvqhv+urNdYeWi7GbF1ztbyvDesKpD6rBIH7Qvvy3qqfNWyBsRa8vG5vaonxTOqYpLua6QXB7FVbLB3yPcLeME/zezuUI4Q4pT6PmEsTubI0tKkhdzoXmM+Y2V+hVLnOa0cibvPWEdS6ifD1zAFSQ1EOjXxhwfGDVjeOlWj/AAachb40POE7lEtHE7FlhnDrhfK6b/cCoNXU08kjaltmtkBOeXvVVQl2DXneOpwPuUmjmUYbKHvaTrhZmfr5qqKuasFRU1NNgMOEd7z6VVMr2upoqC0sVZjAxbO009W5clpg2d2rxxNvd0nZ5kz4tU07WuIle9h73br61Q07WQSR1LsLWvFmSnoUsELo6URRgueeZTMw5u+4JnwhqzPDRxPDeSu2jK4bifrHiE2prNilpz8VpL7EX1j0uWpYcGLwjhvA6POmQNFm2u63AJ+Qa59gewIgtDmuFi08QmUxuaOpvqb/AJt30V2cVFoyvdavjF6d5/8AEMG/+IexC08e79CgGtxN6cKquLmaUqxzfrZq4Zl5KLXjCB9VeD/pVUHPscQNsG5VcTKiJhlhs0PZx6VTRyzRSyQxBj8AyJ6VVftI/wC4KTzezuV9TwgibGPOf8fMdJxeK+TWj+IXUnUE18E0MJlpmhxe0O6OlVFRVStlImwsOrG63YsWBl8vFC8HH9leDj+yjsM+yqXYbcytJyTrNxDG/j1qW5FzOR7U+qrJBGz1v6h0pzJL01Biu2mHjeWePZuVHVRutLTSNfhHEcR6FFobRcENJT1UmrM9Rm2S3DFuA7FSyaVm0RHo4SYZXUlO/HBfoe45dqikpQ4UHhg57sRqn25zjxwqgp7OwNqZZbdY3e1WGd/UnSkDO7vMtbu1m7s4K/ciNtts7MPbfuUmkYhao0dUxzMI8oXHoWzJV23+A/wuO8+1aRbsi2lKwWFulBTeZBVseskDrtdhay/uUgONwDAbvbbp6lDkLG6rOp0f94Uv/OHcrKm3yiosOwD5jS1Pi1UOHztP+VKOpM1erJbC3nKrudrX54R9UKfQv4KrGwQRNk5ebalxOeEf84J+g2wAxxx5yXOLHhxbvo2yv0qpEjYgyN+FhY69+1O7FR+WPYrH9JIW59arLOiNp+Ds+cpOUNeKdrW8lJ5pFtq3nW5Q08jxByhwbrbcetSiDXVPIxr6Z7dxb0BqmnNHir5PikVCdrWvtl5unsWjqM6ThD4YA14NjZ3EblQSumbUtbiaXN8X/lli4JsD5W8nb0c546Cri1uhdHcbi2nR836N+5DTYcfKJWsLRvIusIqZC7dZszCUd4Nzzt605iOIDSsxDsNsVwgpvJQTtIg1EujdmGqjZuhHCT71pYV8kTmRAOp9TfvTMbm4H9Lsr+dQWc8uzyN18IqQ4WSUT4rRnnWxjaUnm9iJ6lo5vjSM1p/iN/mNJVW+TT2PY7/gT/JTuqNqref8o4eSFpY0JfNWtjhiiZj2YgQSXYfRmodIxzTa2R9pTi2pAec1yZSxRSyMia+PWM5pu7F96d2Kk8oewqCnlmZAZXPfI6RwawMab+vL1oYm/BR0zgd1S256FyihGiYH0jg8iGbbkZuIA/5uVxuOd0HNNnNNx2rlL38plfHqbOGGOJl7526LrRVU5grdH6UeLauEbEpyNhw6VrDT7RJ8Y9KkphHhZMGtvvI2lUUk/hIXWP3+deMVziLLOaP0oyts4K53lWe4DrKppZZ42siN2Atc7G7oyzWlKUv0Gytq4zExkTDHNicbcR2rQEmi9MDRnI4Xcobid3524buw+laT5DpJ+jYaOdhLGF1nO3uNr54rEZ9Pcn8hDsVc18eON4aHbWVrKqfFBUTQUeshbqBbjkL9CpK5hma2XFsPeHW9C06Z3l76+KQ4ieiVtvZuR8lvsQjbzpCGDzqngG6KNrB5h8x0iwC7mx6xva3NeUE90xwizRdVj4yXM5Re7SPohU+lKKVnJ3avXvDrOisbHLosqjkrmVNIamSaLe3lLWus8ZdBPqVFNK9kDdI0zmuvltsef+0+pOHLYN3SoqPQWENj8PpB7Lsh6mjxnepRaMbZ+u29IV1VZz3t+gCchfoClk5PobWYL82O6dJp2KlllbDDHHro8WWG/vUtNTSa2jdeWlf9T6Pa37u5d7mtHWd6oaWiDJRSzbdS62GJnEZoedb8xgd/UmVNHhFbGy2B3Nnb0IQ8kg0UwtDnTzvElh1NCiptX+GNMVDSWyVe2yIDxyzcG+1TvptK0jKp21qtmOO/EDoUb530QjdIIwyKTGT7kAOhUFOGhktW7V477nnmn02VNFQUsdJpCpmYJIiC+WGUSBsmZzyIKY6vomvrpnukkcaBz3Ek9OFOrPg8ZME7vjFGYnhj+vMf8618IGaXoamkFRUB0RwY2v524hd8nMflROFvUpdVpGicXM3a0Apu2Ny0lPq2vY2G93HfkeC0nT0lMKiSZodbWhgbvCfTaTdT6uSYvbgJNrtzuqt8TgWGR4Zn4t8lf6jfYtFU9r3nDz2Nz93zKWJ26RhaVJC7IwyOYfSquh0paP8ACDmamWWO8OK3NceCa+MCjlif4OKaBkUnUWjPcq2oxxtMccgsGHE1wBFu1RciipwdG6Khp4i/ZBdLtuKbUzsgkmqjrG6yxwNvlb2+dMFJBC7Fi2mxh2A8PT0oxx0Wka6GJ2Cepo6fFTtdxs7xrIVdFpKZ9HO7XxBlg2Tdia4Ebxx7VM+qngq2OJ1bOSMj1fnG9CtiqamNhlcTDcat29u5MgjmFJVU0muglwXseg9RX/xCrN22uINx85+5RvmpWvqXc27OUTnsv7lPRUdJCdMBt4qWtptU+XrF9/pWuifV01M0d8hxwtja36VoySmTcqhljkYCHB5cHDenVVS6KijgYX6+W4DezrTtLaTD4nTDXmMjaaPEZbptbzlF1ZUU1PpLSrmyTjHtQMJsyPyW7z2KKmgn0c1kQsO+MuevtVM1uA04qW6p8bgQ7ZzQ7FoM/wCsj/uXwle5jcNLUggAWv3iM+0p04mqsOrALJJi5jbcU5kjCyO+y9BwOIOJN79afStp6nSMsXhNRbCztJNk46HjgiqqV3xyCpDIZ4gd1iQRa/EJzZ3Nlp6yVrWmSsEuqlOVrgZA7lWMlbSiOpYI3AS39ydUUOjqdtTHT4Bgmykv/wCyNdpOWWrdVuxMjc46uNvCzVpBlPo+lhBpy7G2K3Hp4JvkBSzkXbRwH0uP/v8AM9JR8Hya0fxC6iiponTVTxh1bBdz24c8upUvxfSDKg0zYjjmdT52td1jdUlJd0wpWluJ5uZCd5KkZ+E4tEUsBDIIpNHmdjrC2O4PtUYZpDReNjQHu5Fzj6VVQ0EtLPU10jYjqo8D2s3utbqClmqp6lhZFF+D2x5Q8MTRbIAC/nTpamCrZDUVMkrXxs37AF/Sjh/DZyyvGbKCN7NLmRrNsNizxHM+tfJ/hI7+Ui+ak+EGPWFzWavIDgDmqt9c+opy7WHBfA+wybH1A2uVo+WjjqOT0skNQ4MvK6AXsT5+hCkon6WGkdK5U9NSv1b3DiQ3ot05KFlRXz6MYxgaIKSTE5gHDGfcFE+sOubTuxtkrZjJZ3TnldTwN5JLRUdSH0+KM7xuJzUnxTRT5XOs6WSDWPf5yVDT18lJS00lO6XFBRMBJuMtxWjaOicH0NNTF5eI8BdITbPIZ2CHYtDu6KyP+4L4RU+prXmaviu6J9m4dVFfipmxU+kI3TbIfJKcI9ajN7OIVdyWmmqqh8DhAxjtx6bItrm8oeyUDVyOs3ccTj13sn1baOsqNHObLTCWBt4+GWLoVDXCCSNujyXsbLUvl1jt2YJw2CGxGc+EYyUkba/QVDBSvtHHWl8Rkdbn7O8Z9KhENfoYuaxofhqZHNxcbDHuUNNFNBUPq6kY9XIbYQO05XsmH6i0jWH8/NgHY35nR1QGVTDhPWWn/Kr9IFvxmJjIWO+iCM1xPc49yamlGxK3DcDMIPfUzOgabiLxQo4IW6uKJuFjRwRzOQRyZl1Lcz0K2x9lcorWPE58eI4T0KWPR0ZGu58j9p7vOqqthbynSNY68tVUDHLbg0dDepRSUYhgqHTBmtIxYwfq8LKm0nLK6uq3Nu90pvqjxAbuCrx0yFP8tU43YaM+e5/wtVomHlU+jaksmLiGYMt2apvwlCKflB1ce2HXPmVC+KkadTM2Q3laMgbrS0x2XS1pu36BDGNt6lSRjPFJcqMKk8qyNQ5pjc43Nv8AntUNFSt1dPALNaO7Zwa4dYusoYPNGEaarp2yQv3jd61hFJk0Zd8KFLQ94hBJw7968N/SF4X+kLw39IUT3Zuc3P8AL0tUBnTT28zlpI/rGf2/kWy6/WU0vNkA3dRWc/qVzUH0L5Q70Lw5QaJ33cbADijOX6wxuc9rjxsf8Kphvstfl581XAHNsj7hSeWof3NfCUf/AHH3BaF6qoJq0uOjSEnuVH/EmKi8sfjc4NubAncE6N5DXsNiMO5c9v2VzmfYXPZ9hc+P7CbFG+PE76m5QC99gZ9P5fSUY3tj1g/hz9yrz0yj+38Yt4AXTu+RkD9WsLJYQf2S5BUYZIJAfMsG+J/MKHX3L8VpHS7snBvJ6bqc7j6LqOGx2aUZDrap4/pBnvCqgHWs+VSh8DDeUjE02J+9Ux/0a+FH7/7loj95amrTMT8nmtfKOthtmoHQNmnZHcPfHGXNHnUe8XHEWuqIBkzyHgnDE42Cuw4hf8WUccOSp69n5xgx9zG/JqD2c127uVVY7wj24WpjPoi35eWJ2bZGlpWlad3OgnwHtFx+NN5KlaCWk3FxvCijc98pZlifvcqeQfpWD0ov8aPMIxsk1br791+q6Gvljcxt92bndpTIx4xWhNGNydVzYiOnxR7VVfRYzh1BPtcCzf7itITEtFp5mjrU+N5bC+fmvbdpVF+5r4Vfvw9i0V1VLUxaRjItye9j043H/wBKIzykXwe/a+8KDsVURvDnkLws1/KVjLJ9ood9k+0V4WX7RR77P9soRvtdrU5vQi1mEnf1KNjrYgOCaz0qigH56ojb6/mOn22s2WWOYfxN+/8AGm6mKaTCcLSVrGtOG6jlG4Sx+1YT40QRbG8xOa+++1+pXlne3LIMkcfPmoQ7aPE9K0HQDNlM6PL+paRfcDvZCLsNwA256FWyX3TS3Ft6dIJjgx3dG/NqouFqUhfCn99HsWjP3lqjWmBfaIYG5b9qRO/aL4P/ALX3hU/kqdu8GR3nUDtW7k1YzHEcZyPFqyj/AKivB/1Fcz+opsTIy+SU4WtDt6bCXtHIw5pbvve3ssjhqCwNyyatXyqXC0XDcFgM8/SoWiofjgu1rsB3HKy1nK35iy0LTOdjIe55PY0/MTPwqIg09rT/AJ/GqP2aqh2qXi3pCc76JjPrVK76UIThHM+DauXMGavLPjaB9K+L1LWEc3NSyb20+scPMMKlaPzjrJ9e5log4hpPTZVzwRfWycL8U7E5gJduC0cXYdqAr4UfvY9i0d+8NUfYFVMdPUMYyoL8DDYE559Kk/aLQB/Xe8Kn7FL5ZRb+cpziaejNdK8ZbnJ9ZOM4ub1Kqz3vd7Vi6+6CqNv6Gmkf6wPmM02rfI6EYrMF3L5WwdtwvlTPWvlcfrXyyNfK2etPhpWNqtbGbvxWwKRjxSYHjNxebp9PS8mkB8Z5IVRoueljxMsHSNkvdUEnTGAhq7YsV74y23oR1tUZvq4d3n3qSb6LSVpKuncQC0tBtfMlU8cLicDnuOVvEKfFOXA6xrt1/FCqhfBBeUgDxs1K0OszW8Mlouc5sbR3Nt/OXwgmFwJalrvUqD9u1ReSFVROjkfje52z2lPPS+/qWgD+v+5QeSpYBTSyjfcblVN1bonlxFnKWPiwrnBcFLNsiNwNxxUznsfGTIQMXjDgVi7gQKqj+hpGt9Lr+75ibb0TPTwSX+lEF8jpP9kL5HS/7IXyKl/2QvkVL/shUbKeKOFr6cuIY3DfaR7ER1rSLf1bnehqpb8B7gg1sxiLXYuO31GxCYJajWhuQAbb0qtl3Wgd7FpuSeJkojDAMbb9KoI4RSwVEkjyWgAOw4D71UB7aWeoZK0kFoLgMAWloG7NqidoAGXOU1rulMmWY9ioOWWl5Jo0yFmEWdkTb1LTVZR0lOyCrna/VyT4TEbbsgqMVLBTWlBGqOuv7FC6OGAtwjN82E7uxOil1T5aoOnuw3DM+b61VNDmm0/DhvWgDib4dQdimN+ZM5qnj+mwFSHg7uAdJVQeiMqNv0oWORRCagtNyE7jEz1H714ZiykYVtbulXBBHzPR2rwOPJDcE/WRDoWi4tz1lG13a5VusYwB1PNuv+jKhHFlvYmvqZXQiF+JjmvwnEmmKeaU9DnFVvDFFhWmpv0lU1noCrKyZsMRo8dE0RjnsDyqOup9S81L2UDhI3JsbngnzrTcz8QbHWyWI47RyQkzDmyY7jioDvvo6bI7vBvUzaPkhY8XcJI73TIq6jozqzcGO7E2np6LRzGMFhdpcqmo0uyOWCkonYWU0eG221VmN7dIxV0/KYXgiMsa7MNt51oiqmLaMQzXwnvhd6FAY6TG0tuDrQFU6OmFqnEag8RY9ag3WkiIPSi/uNClb9PCz0kKi+vT29DkU5M7ldHTMvrZi8vJsxvDNWaWVU/TLIGsHYEWO5G0NNthl8SAlbl0gYU2oiOHFv6Hdv3q42Xt5zej8ofxo+N6Jv8AcVI+npRUyDxMhlxRMkBiJPNfvCczpjeLfy3JgblsD3oa0OOpONmFxbY+ZNx66+6+ueD7U6MZDYZvutPVOY2ql3oYnu2S8nfxKqubrWvBaeIVRJitdzHE26giQfGG5Uv7hN/5TlJ5Hd08GguLdFPOXlsTWRaS0aXPaHBus5uW4rRzpHwVEUbrN1O0fOUGNrqSJr2B7A+TNu7LqVZW11UamlfAI2TRR7OLI29S0dyJ8jtW7vmJmHJPfrHXdwVi91+i6Gdmqjg/SVMTPWtESfSMkfsK8yKanP8AosJWoa7vR23M6Xb7lWL3tl4FS6xuGppCMf12dKeaotGWV1ip3axtNIWllriZvQmSwv374zvj/wAJsg3Oz+Y0/wC5D+8qTW8rw5H4vz0ZYeU53aTKTdyjb9IH/wAty8ke8pwbKDiGWSb3wb1T6twc2SUWPpVY+AgPqp5YiSL2D3YT6lhFS+xP0FVwuqZNrPmJzSXS5MzdldP65Amfucw/6LkfJ7jl8Jfq6GmP9TELxROIaN7AUSYID/LCeOTwMcG49lgVOW73TcBuyKY5riWhzc1idnYbJBViyMkbrvW03+paDh+nV39AWjZfoVdvSFdxAAbvXPZ6UBrGX7VUeRZHC6MPf6lG8273uINw5VTj3sviEHnQaK1jKJ8WIu3kHoXJWd8dMQy/vU9RTG2A5g7uxcnlyexxZn0j/HzGlP8Ao/8AvKkfyoUe4a0i9lnUNqs+e0WBVL1u/wCwotEobgu22J4N8R6FK4zN5uzaR+/0dCj+s7gXElURJwwmQ4Rd/DqO5CT9dj3frVrHsw/RI5rlUPsGkNxb96c5oIybv3hOd+sHWqYvt3yknI7NU5O7D3HL4XfV0JN/cxanNrREHmx3lSMdtWdhHWnCQ4jIzJ27JU/VMPOLFPc3wbQ09W9NODezpK5sQwEo6xgy61oKOMW8JIc+hNc0XMdQxw9adHPR7L9lu2NoJ2OgjJfnk/Mr/wCXMFr4H4slqzSloe4NxYxZPdlhYqepDu91TsMrei6qYI5PjdLNqznznDcouWDX6x7mODt7SOKZWxHwZDk99O9p5WMQ96q9Z4Vuql9vuV+n5hQn/SH+5Pe+anhblczMxtd1WXe5aeTM+BFgPMqHyx/aVIyOPEy53Nv+cPuRbyeci21kmu1L+fmOpUzHRPjte11Bh+g13/UTu9t1ZIyObHBV73FjGNhI2RcWt0IuYdksFjuWX0rrRTARd1JK30xuTx+Bp6pmYD6Yh4ctVXU0lLLxY+2IeZGSh0TV1sZ8aGzrL4U6U0tSciiqdFSwxxueC8nI8OxMDMpRCC+Tt4J7DvabuPvTziJfhyytl1Kktv5SPYVUxc4aoHrUGRsWDigKKTVRu37S1lVtSX333qhcGjvNM/zXVV9Qtd/UFBamZURNF+fhLSn/ABI7e867NNvTR8nHHGcSgilY2O0lgGm+SO7NQQvyhgGsd7lUyUmFmsN3D6T+lNFSLTwyO1g7eK+MVElLQA797peoBYmGpkdELtbK7nHsVZLfFiDGYrbzmfemdnzCgP8Apz/cnQ6u8IAvipnPH2k8wQtDg62xSnaWi8iNZIzzZFVUM0jYZGSvaQ42ttlSBkjHG3AprntLW9JVJFBI2ZzYsww4rKCGVuEiniu0+WFMyK1W1viObm1aUa7C3vLrn6OXFNOFj2mzbbwixrDiB3LR0Fi0tjkGf7NO0JoqV0MpHf5WGzuwdCL3uc97t5JzKjrNG1c9JPGb3Y7ndRHFaTqSBHWw0kjKlg6cJzHamyWuHxNabb72Uz92sy7FEN2DPPj2KNuDfOLdW9SxHbLo7h3EqJuJpw5HPcoHsGLsCZcWzTm8Y6Xo61pBjBidqiQB1ZpuJ7BcDim6vb8lXlszylCyFwkIdfIqPXQsqopRsTMOFyp54r44Noj6TVFK8XjlYHKkjgcWR1J2x1Kny7yzLDwtwUZpYhdjTkooG+Ecbu7Uxv0Rb5ho4yOwAwubfzqLUSyyxObkTUgYj6EWulwhxLS4VNiOJt1rRTI53T2nbtOfiJK0oZIbudOSSCRdbLZm/wA0oX15/mlG1O05eMSVVdUUf9zUL7xncKqxMaSYyhEHg2c0i/FStcxrhisQ5Urm52imd/0Sq2rlJLpZXb+Hd0/QjOOq0VM4josFSTXu4szbx3J9vGdeyjwXJBvu3JjG54Kptj5imPda+6wWkqqSor4pZK2YHVTYW5O6LKLUaR0rt/Smafcmyy1+li52/wCMW9y0zHDJVSDvfhpTJwVT+yd7E0uZm4cHELZM7f5hW1rnfzCmmKNxIubYiSbJrdHzM1sZvqH5ehTU3InXbdrc9seZcjlieyqgOxibbG3ioZ545mwxd7xas4RfrWWHVjdITs2T5pHYifGOS19u9R7uv5jomUxtlxF7bO4bitrR7LtGXfDkj8SicC6+blopkNNBANcDsdhJWkh+t93cCPkqs/Ys9o7lR5Cx7TXC2YWtjkItxvzf8LR0ri0iaCTMfsiFWQOY4UlU8z0r+DmE3t5j3dO/CGojLGVdI+mpcQ5zbbTvTYelU81jqxAG+eylduDzksupHVj/AMQyx9KjH1lpD9/n/vVOo1pn+V/aqr9k72KPye7o7rmsvwhocimqr4nMGTXqOPSUTeUw7neMCnSRybTuLheywyPiczoLLq5ONw3D/CzBipRv60I2CwHzGnZjMUkEmNrt/BYBUNPmV2zNaexRaSqZdc6JpDRusTxVTVRyBzag3wkbluCHNVi4NuFUUTTtPhDQfQuey3YpNtoy6FrDK8OPUi3WKmrZNrk7Htb5wnUumafXxb2YfCRnpaU/8HVbuTX2BUMu8DzJsnwg5RWhpu2OPZiPlcSnU0TWspzEY2tYLBrbWyUdOy0jI24Wk5FeCb6VzWN86ayowvaHYrJuCNrbO4Kt1WqMUlS+QB2/NR2ZCzB9ZNhMNPZvHGVVVZA11SGh1uoKVn02EK0Za9o3X3rONq5jPSqWWZwDYX4rN49zbaL9PFDUk9jluaB2rHMwyO9SAbYAcPmQH5DD1LcjktyxWQPaiTn0LctyLOHD8Z3WfxD+IPxsv/rL/8QAKhABAAICAQMEAgMBAQEBAQAAAQARITFBUWFxgZGhscHwEDDRQOEg8WD/2gAIAQEAAT8h/wD7BJtJRZf8X/Fy/wD6uX/N/wD2X/Ny5f8ANy5cuXL/AOLN5i5VZa3EC6/8PCgFNoPxY+IOOToD9TPxAA//AChgysLU6bjAC5lAnGUlWcJGOoS5/H+0W6E2Vji4z8h5itRhnFFdWaYJtLlMC4yk6rkRU1Zh952JXECb6KfuH+zoWBf9hf8Axf7Mu9KV5B694/T5Yf7OGnOG31nwiuveF78uGz1mND2HWusetyxePWXIGfO6vrMBDx/2CY4wdaoJkQbFlhc0uvWVvA5qh9sFzkagUWTkOF+bleW6KFnqy7sC4C44CKPNCdg3aARcNhivLf8AwWCr0z1NQQrHCBWuY3RYLhXHxDKEo9Irn6i9NLXRvvKFpV4QS4lpD0s3yv6jt+q+Mc2DPWBSyA9WYyT0P0RUr9pMEvJo1hOBk+LlKzKaF07lRQ8QoSlWPSnmACkVibL6tqzDICDSi7iaWmvciBKcoIx2ZwtXK1qXzBV/NMn3gz3gBma+VGpcBRgvOkU+UyXDtHTmDRjPsEI+qPqO/WXBl94abniZ+P8AE3HAoN9ZgduC9GyUsjLfo/2doyxAllYwJwjXMatEzc47TCdKVC3mj/yW/RcTTTB1x/F/2uo3tx+M6BzCjLXclhAFejTKZAycP2ufhN4BIt7KJ7sQSGaKhQtrW7lXYP3JubLvcjDqUq8NJyAz7Mf59e8JbSsseMGikAasvCJSovx9QL7jheTMe+I8BCpLbTY57NdAJcbkutKuyqq5FMarpKxRi4lzbkMvS9XAAOm+8MrzE5bnC7WXIQ2Rv25LReNdPeIw1ay2wzDuKVzHVVkDRCl+77mXpqRhh+SWXtAZ8I8uPW4ycn3FVAKLWLPKofncZiBX6S8eQMF0oSkeooo8paM+Gogw3Ns2DrvEyICUJ7g6gR0Etu9RtdReEg7TlC09NAMyD22dsd3ieX8XPWOZx/UdfQN4ETxFBYbGuePylBIxzITOb30ZjNPUDuln1YifUbPpKERXAJfmNaXXbhge7mFnuOa5IE0tMrZ5hNTSOgF51WPeZ+C40P8Ao/EycBnI8O+IsmxjdDd5XLGfp2NdFME9MNXaeczIIXiTXJ6PvEUkiqvMKFKV4tiAI7dlqAAG6buJdSNK3V/+RSvFBdCOK5/EsGCdwcF705uGe1QBfrierxrhlODcVCkGhzVwbuQVDtmByx15u8sHLfMcTA4ClKqdJTyG1fsf3UTyKRiLznxKqEA6c/2XCOeCMXUv3BCwKMG5pgYrizfiAyiAvTCOag8Si8DVblkIeAQJXTvKUqa46uzXxK6BCyCvLvM59sxH5yjGj1V+0Cu5Zv4/GYG4t/0kTW5rbQalZ/sxHuWvB+Z1uJop3aFhiuiKookKBxrXV8w0fd2/mczyti85jLlwpd9u8QuoUDRjIFoxPrdXMPLozMzxxLqXhSVfmCGTTCwI0uC9y1Cghc0ejwinAhctoNuMJpt/Uwy8fWQ8dLjS4Ospzz7nMoUYt2fmExWXb9oUcyx9CzXYczNAONC7G3uTB8RtAui3I84ztiEaa0XpCXIDFvzK6Twz/MqZBNa8+WMa1A9UGWm17UZjVuKHrDo5iY7S7W5Q9IgwPthQkS55g44A4FKGNry0Xux4xD4VqTIWLqNi097JHLNRcks8v08SGRwEvP8AZWzZemTFv8JoJ1RV3ucAtqdY2UmXV7LEotruJuAQlCF2HgVepHlHa1Y+JbtWU77xcIqwA+BCWpTKmDkzAZzMpMKLU479IlB9jYyOBVeqj70UxRgOuu09Vv1HEo4fIkydjTbtVh3vM0GXfrOZng9MzMVukqANKpu3tKYIbXEfO12uoRxty3QBlgcp12DBOna3p2hpbAORe7r2YtawBS4mV66UeHFaz3iZ7FwAic8XxCSsgDhadn3F7qyCHhimIOpS3GUPl2RqHqoSheGXrCVnXt9qhuGAEWC4DxABq3JsoaviZ4JC0TogzpKc1YFfM9y/zOaYxaRT1lp81KqOP7PmMQFPqXTrHiSrzoJxtjAUVmfHyrS3fG9ktMSzTUwu5LnDlD0eCpYdY0GCwPRmzdMxhclNS1XvQXMPFXnlaTdqDlAO0jhJbJtVCj9kzZ7rxKX1Ly3xDpbCgGzVb8QI8I9Lo9WdF7fUOP8AS5dBTNWDd7RU/npwFGLkTEo7wped5DExbuOLUK9cm89dxBRpp1zqAtbzWpvGQKHROT2Eea/eBpYAZluICUxuutL78fUDqhOjNZ94jjlU7Y9EVQCwfUdpkNoIpt3frMungbTllkbHn4hf4mRbU3KZlwEaa31zK4E1g8pS1t2WzFY1AdvROFZrE1Gx9Zh4n7r1lSTHKMsX6MP8H9xF7cHUSPiXs1f+TEnrpiYYgEJQ2udy3E1hyo9HPeY3wHmVWABUE7TpMW0Wpq12cSjp6CQrl9vK3fbnMzeLVdCuisx6KHAsYcMQpy1cIpt4xtxpqIYdfM3r2Xrs7mUXhs7OX6nJSUOjZ8z9M5nRIM6t+JjpT0Xqes/EJJ3CvFdWAMWOhAkNZHftvkyMs3kjj2Qt9iZCOFDrjaKtCAimU8KYbLicVmmKW+QtTxNtViFD3K+PEtaKUquvKq8kxj0VzDaJBOrlmVjMKVHS6qIY0sVRoW4CFdnEpHC82TKi2ztu8Vg6lTa+10wrqivqDbyRuRwnLEcI5rWjxfhLfUfZls6mxhuXb/b+45CUIV8y3ySrzSirqll6rxYZTC+bPCLy89c9jGO0iXJbcXpRWzvlHGws+0tGMLraSynO8kYFZ5KXU7ulN4Av6idz2LUVgYPaNkUXeOmZf0S7idF3zn7JeKYzga44ESuKfRj+UrS0rICve+OZURuHFYWmxwYN2QPAUTt7WK4z1mYlGMmAfOessbBsKxO1jHvMDca75sVSkuLgaO62BZ41URBhpagXv7yzUtcxmIJAy5W3dxllqWzCZQRpycx59K2kX7x43dvEZcZS9EXGE8tvZj4IBRBhvvGDRg/vFqG6iDVEVhAvhApcR0VLwQMHYhM1RKCU67218f31GoR83fUOvNM5u/6hOeKPeFzG5mA5VB5rkNHvAJniunGoZ1Vsb0z0mlEHd3xCoENjO5TtDiGigXrc3Ayc4YOQLdDM02kM6CZ3mII5C6+8ojms3u3sKs+4T/KwaFHkqrNAiV3W9vumzai3oL98TFhjd7IYvoj5vMvI9vuVU0NY4q+D3iMnLVTWZmKdNTAmoSkZ7BtvtXMyTmKNMbNGwlxzazsWXdfEpcq29JUAzPGDVr8RqISBOrY8P+RB8ei+CVsxizVwB0LCN2zg95WIquNXS/ScwxKwMY9I3KsOTMekrANValuuUahODrHndI8u0vhOhMY3h5TCtLJ3pf8AfUFkj2FPup66nkdJYODnCrCMGZzuLwMMoKJGeOYzwhdG/wB1GwNKVjjXsTlhe58oJEAugtc92WRK3Y64OQovMrll62MPQZfpuTbG7SswtMk0OxNZhq3uZGZjRRGDrmpcuVdCiar2GqzGXhnBAArto7jnmXWALHkVj2XrXL17znF4FhgPQ+63mdOBH4PJZ4yxDnFhGLi9yGHuuTnoEfstBTyNdWWKwI54IXpLA2uOrSCimPM+WaUokxtTlg95wRRsmTj3i7osUsnpBBInBSeJ1ANhmhx6MSmC6Bmu9+kwDeSxusVfELsVUOY/oXGZjmULvC9qLONKD++pC9U5RK+p2/Ir4oq5HBKNWHtut6l7dGNZg09Is1jSebtNRt6fZtG1AXnuDSpo1iAXWoAW4G3beoEfDAMBpRYKT3qCJjthhV5x8yhdbMQ+2+dYmo1XJ1V+vTtEQoHaiqggyat42swVC5i8AJ6HjmHYl2/Tbca5gxV66uBNuTz5HcTvh7TGDRxQqu+5YdqrC8EG6sp1s5yFcg94c29Jxt6r6c2X3prxRhDuaEyvxNzKxefWDwxqIEouBzSLNdzLVhKi80aOSC51Aqaz47w2kuKf5OSkgtnOp1s/bpMStUl7zp6TU1tVSsqeWLhYkJgyo7ym7TUPS/VPYYFfZ+iofKQP7xosI6iUzAh5MT8Ie/jPLS1EKeYlX7GeeuIZ9c7II094z2KHBqz5z3I4DUE7gZ+p/mcCb9UvF9I0h1p7b64RNTlqavJST0XqKtuQM+A300QQPBZ63SAMmKL9EwmLfUXmo/4TLWCDo+G8S2VIJ2V9PSZipqKzjmrrmXzHk3Dce565ejMubs8jsuu/ovEp68X6Anr8IaXi/B0er+ZXFmVeA+WVXAJZK0CtmT3nR4ldTTpaqYgaX4jnynnKIzlxRQoxqGO8veJdUfsasZ8f16xVCBfX5iNmv3uPz1w4Tj3EjRBDsGO8JQKZfwghXgNPhF7Esc9GG5/4BsSYIoI9QPtcx/UjDEKlcCjKianlpaxpBoPe947Sr/BKv8EUS8HhEkNUCW7+8zqQsAwKUk8T4bAecwj9KDbdLa7TaYX4rpLf2O8qA9BATymBCAuIVoo9QhejKhq1KmHAFw0KvZOc3Yg0YqcKtq3sR4iSp7C784ADoVylAa9j0XT7Eeo7LccHtM9XpLAcJiUhkP7OlwV6BlEqmZADuCkqWLgKdFlzXA0rtKdtkZ840VUrWLxMfIz4/wBifDlbFRQAjzbfniUGZhIzphKNQds7fENUNiCvmPpMbxLQyVuyPtf+FTjZPX/CJ5DxrFHGuiGzfCL+iHGuuAp1HCfKaxSaXAaVk2RkFXbIt9jPnPqBoUt8pjS1qM9nHV8zP0tPkLzvcHLCicorqtpnhVKLlrh+nmXYgXlfJY5pE1ipnBobPoKeLyghmFq2QvQWq7R91YHZqiu/KoErDTX1L53VpVmxumc1uoZ0/ZNchhGInWmLAraOGl+YG5qKGuRlC/WpgttAK4q98RNBxGljPMMeSJGR9ek+7PYz8k+DLgXJlgw8tL6MwpFDm4MVg2hJVDgKqUSKMK/1w77Qk9bfCdkLS5Ck9xT6pHEP72CyBugJ9kU3Vr6io28I9j/IcJyefmdonIWtKvqttFdKQVm1kKVY7vxc3crBchW+ByeZ8v8AUVpepRtsF9BvpakWZMbb2heZT0Em9lILV3FBQQ9CUXCc4GphP6nuAXta3Vg7Dz5aSODluusPjVl77vOryQFwRemIE7b7xx47SqNnWOGnyaqPUV2rE1DkRsWb1uuGZkTJxFuB4E4AthbjE3oEScTdKvFRg7RMGFi+Lsw85G/VhAZJn2RiXQqnvBf7dk+Gg/toqNIRJn5UYtp42V54lQsGbwd6XjzKE1yW2yPhkZnqnxmZigeuH5gW16YBAz/wfL44f4gv9dRDK4jBicACEd2MmNcAB3Cp8y7Lkugw9FEeZ1o7oxF92BZWVBV7xE1bOK9Xq+lzEgXLh2d0XyABFaJQXRMVzLU4KvbrDm6ste8MS7ZzDI14tHPox5ye1EXBZZI4Ww51ZbUMfdfMwmjUdgxH95hlN3hHT3qWwhrU9a72hjMuH4daEGM6KtYvcRmODcXWyfTmNgyTtN4MMef4DBW5LYUK9OW3z/0EQbPpHmI3Btu+xHfMp3FYm4qBT3KTeGhq6JE25JmD8FlhhBUQz++ijLVYZ2ZH1ifGdSDF2YlCbXs5m5dktU4szMydpRH5rNvqJg9f0JoxA9MmKk6H/Cal+uBUxbMvCIE9VoKwKLmuFxDXsCoc3kZ3ObzKDduVKTOD3qV5Ds+6Zy9cx1wLlgkhwXtgz2R2IMMU7BoV2sEwNonnXEbLrptmotOzRcw4AeOiLF+uRvGC6GITqFUI0NOpdwiueAmI+bpqMeKlSJkp1+mPl+mljos8OOjUxVRzSzplctdHESJwIMZq1Tpmqd3Gakk9w2ZHDMBZE8bwC0aOZsMtY8H1dHXmC8PdTypg0XarEbllEmsnU3lMs4syHcqd3xU9JImk/wAUKozLrU3W7SLpIvLlQdNY8EXgmE3ZeFOIYqHi1iq+IaoMBd4BBff0uE8JQlUtMGT7jtYsotMDoK6HWXd5CwyapeyEp0Slwxrd9GiV2myEayaL34qJHZlBQsoZc2w5N/4xr0MPSofB/wDC/wC4Uqcemr6B/qUS1kZ7ToRdy1UuaslBkb7x02ybeY5tVlGmH4qUg7aA1HOSIcQW04W5qUGUByS8WQXG5kuMGFaH3CUtyxiEA8GkrqOIPlLUQmoXo1vsfIcwVX2h/wCpsnyyFzNrC/WAIe6rls71BvBwQGaBOhwXVd98fQxoGUBYLW0tuOt7uGodop4OkVL2t41fgo8SuMCVY8JdLeeZuSBPkKfSIPIiqG2PPSZ02gQ1snADAw34c7pWFtEfGGmlM1fHMRtxm2cldvacPFuJnB5mbyU/c2ahQWlxC1XY1xLRz+5FWSY1uImXhqUVRsObqYVwvuTjXiC+cQ1RVTaCMTyQQDQVa67QWH51V6qeYehOtMJWDIP2P9WXLl/33UwcJb9DAhzmombWkNXvyQVEGEfE2J8E2MkB2UmUvvtnYVxAaDYgBLCWI/Ex7oIuUN0PwHEXiYLoB5cB7RbSx1RqNgwTXJcXRjrmA4JDINY+ouekG66zV2WG+hH70/BF7r7lCRgNZ9uDYrLvGsS4NMPD9qgW5dh31LxOcaNaiJ5zsRLqGLxBP0BUtI1+aUsotz0wg+6vE+8jeW94RgDhKHhwVfeWBPnIGNJ4Sz9mSFsioNMEcHOqrcuZjdR8Fyz7bgr6b9CI2AK6/wB25lgEuwV9hDfUh7QrZ2wjBMIzAjlnUSAMBvcujWGdC56P8jHIZjFA3PD14EMqPEXpfDr3S9qxTgAD9qWUOsRVQajdUe2YvVH2zaRVs95ZL9IzLsyr9zI7Vt9SvthMYcytfefMsXLYWuKYrALK6l4GSFqPQs7/APuX9r++seAPjX/Z1t6f9j0tznEVlcy0cP8Aof3Eohbewok/0zWXP8Vj+FygMNXz0lM0FW2YnYOHpjLAsVh7x5ivPI7fSO+1ejuzDHip5R5d7djXuWRL4xRm7H5e87uBLN+h0JrKmno14mlG7He0/CWY3p7yo8B+sVP+1Mz8ZFgkfwQHw9MS6PcvvYTLFIFy90wKiWepglBuFfR/hfpL1qKNIfjYruZJXbJKctbl7cYsg5QinXkX8VBlW94vB9ykNH7T+64KtCOolMHmhh0/wJdwqJhxLIl6mfUlzkT7zkhzRO++Yg1IzxYlOLqbsxYRlbNJ0MnpAnHJLZ1YGutZiQLsh6Gv6q+zAQsplNFHBMvofnPufiU0BJ3pqErsqx72OrliXeRuw3qWeY+MWf8AQMftn0RDtB0a2fb7wl2n4WJn7I53F2l4zHrOLwF5O858G2BB55eTYLS0tMaMRrrehekfgVVjLVQgV08zOBN2w+JnC8lO0MqrHxm/BNP7vSCz0lO+QnZ8GFXmNCGXaVzszPcW/MyMvL6zEp5VvJEAbuOfbnxKRYCIHqTPtG1FYtx6n6ROqk9hmFTADwxb/EYF0ivLA2iiGKC9+s0XFJyJzL1ZULq6dJQqp9hPKPpR5On0Ms8R9Ee9LIwGQHzxF4NfueBmTvwPxKLgsA9THtQE0c53GCkKlVoyJwSNyRBacTXCnIQEKMYoo1k4NuI9aNCoW8uyWEadyh6Lw9ou0RYyppXmphu71WOPsnP/AAMPjFHIr4ftE5yQ03MkBg6dI8VzcgabGzEHWazRP0q0hj5PU4jFtQy5T+dwM5AS9uM1SvdlFoIyl6cUCVAdzl5ozmC5MaMAbjX82z+aRvWjR3jKAFqkazOgo1xi3T6o7/SwRqIOclomvsSzd79wePkH7X8QRwX2V8uq1KM3TF3BWr3QXTEvgavy6yrZf7UsqYygolruzPSOh531F9sMv/BQ1JcG5o5xGJY7KO2pWL+5/wCIH9H1Gj/X/J+lfiVZgWTwrUscyKBfpA30U+wGoQKCJcZuqqoR98pxCg8o8le3iVxSbioGXjKeCB0AXI/xGJ59YqCxKAHYZE5JcQ48vvNCsl71fWZ/+LKzT8w9wxfRjY6SV/uYIumvOGOdIQ90Zdp+4+G+pbbQd644KMvJMOUwG5nHxtwcmy9/EtksUdDDVogr8Iz15DYd5UO5MYcR9L9feBLl/wAP9mFYsGNRofspfeWMFFRbNf4VFJYcALHidv8A8AxpbofuRX5VBC4V1WBrNgrwkuDNU2zyVWtH5nIR/wC/WVoK4Ique0sYAjUBQ40lKpMy3Fmrs8wANauACo+JUNGhO24xDfJum7HozhmUEJ2aLRO+I6Q7sg6UJi52TocC47xD8K1Q8j9Ew4WMKwpnPaBuY6vGSW43DXpM3F2nZh27rfuKftluIFwSvcKPiVfg+2PxNGnZZnkiS7doJYBb1ho2XQh59biewnQyQupcjLs/s1OCZNTtzwjUjhKggw4nHXrEoWQX/ET3zmJXR0XXSGXxLhxUT9doCnK3ewVvxD5Fpaxe6Zg7b5iEob9Lf+YybuCSLfMzEPUoI1HbyL5sye8NJSlA889PeEYURTIE1GKD5qCygY1zNdZbxAtGur2jypg3cby53zCpJXF5FHXdCx0LNPxxqCckZtK6VL6BGGlysOcnEdFWi6rolTmTMsIzy9P0SChxsfqveelbnqBlW4uUXj8S2Np6JsvSZ2Yz2bi2+rMIuLqt1MDLxcUoFKv6YhPwJ23/AGd5X9T/APCPcv4GAvhYZzIr0mKr3E+o+NNAPtYgmCu+JGQC1O+EN2r5SqrtdIYh2xVF6stUeQSi1b151GzRnjk1SxMme+6y8vS4zcMqppgs2pcTJNd9yRRZzNpDr+U+icGUaJe1F76HELygs7DcGpjXAK4KfNd9tQxWgm21ugT5lzx2HUAA4XtLJhwpl894PLqNwV09vmOjMqNLzKLgJmoBbfqGz9Tu6D6QBe8erv44jte0Ee5IiMP6ihwd53nKaZnQKgwmq+sWSWLq9Y4Vk9L8G9eZXtAPNOH7e80Wvo/qdTKNv/lH4zHGh6DhrcusGUOZyzdhC7p+JTK4vvDk8J7QVvpgq+pU+rwPWI1BpzRGyAtVgHWAdRtiyU8BMBxBpjYaCshC4q4L6ShrdkzD1H+5j6Z4cS225nQUo3F13JW/Pf8AaLeujkntuKljNaPFKkChN5cuYSDdoDMsHm8yrhlbZiMraanNsqfi9Ck2kYrRM8ZM1U+Jke0Ap5l85WPWZquheUa6TAZa7ZcF9YxU9Q1wdjros8Os6shCsX/qVNjHH6+EsGrJtVfyh/q3/DGUZYY4OskDRoVUa6Vz4iCpWsuR0MSv9nPeLKhy+BGmpVyvDTe7gdnMtGsyb400ZjFrCs4ZdAiuRpWFmoOlVm2HDwxSFoq4D29puQvCvRidttKF0cn1EVWAvSEdTj80GPiX30hE3DPZcTAtXMzdyh3GXwOgGyUGWUiu4MofnqODhDPyh9LzFospFrY8REaLAKFZrSE6Ed6sUYH2QuSidGK8563LZVyyl85zz7SrXkg4uEuWx0DiWZqa4g58wZVIHQwniCXFKUs0vMa2skztUesv911uyycoMWassebuX7C7qrLR81gmOF/1pcr+TPdgJtQcZZX6z+IdKLBRb9ZHXdf5M3UYsVpp25Q/Ouw8dJdPogGR3nzdQiKF1RD6i0VJ7sy+kpDznHpDVtcCFuAqOpOcdFCgs65uc1HawICUvpDevZm/clDbwkJ3spN5ABDypxDdKJ0bUTQHXzNU5ZwMU5mPfzhG5vS+bC4othhO6EoeRUPG3aMh1gwTd9Dt7jMHX0/imjCzyl/hBTyF6Q3MntnNnUqJSu2LB7XKMq1x9L161K6OJT5LGK1qlxbER3+B7y5svwveer8QDq4nY7PG4/w9yuH6llu0LXy2dBtmyeC6h9qS99T/AHX/AAfI1xy21YepavzK9kAnbPJ3zLoazSm26GRhukYIq14gssYLOMjnebEXmozXRolr+ZusJwbjGM9yPRwBOpdF4iCcPWwEMLDudBue2ZgV+DebuUzzU51bfdW7j+Au+Tuu5prEAHOgOzuakFFntrKLiDLMEJR3JcrioLw6zCTU+lqoBVNeuLVlM/JIOoVB/W6lqWYN3mikw+RTCFaJXQvS+uY5Vd2L/wCE6xqj4hK89ripB6qpbq6qYu3E9Ind11hd6k30+9OYefOJhU/2UZklqmWNdXC4i0Km0KwxhxN+MD5WV/EvnYr0/wCCmqz3JqpVBt6SpTaqiLBwaPQcB9pwkmvBH2lrNq22XTDXpU/mQnj1birfqzthcMosASmU+MWh0YpbXRTbOIwwgBxn4ZTfDgzrqx8QfYFx/G5dZwiK8fqC6uA7Y6j/AJt/pzklAyurTUCjmm4/FFoVNU7wPK3iEALKmMa+fyM8EtgD7CPuNlcN8XrKxHmEUemcZaFVufZhX0CV0fmlXXpFEXUMNZzDyLh7aiqa+NLd9bH9oFi5p5I8NbsaFORAMB1QdVzFvDRlXQ6TIoT9/X9zK/4CPiaYpUneWHAdILDxBTD/APks/oAziR6gPhFhNooV/WYLh9sH3S/MPN3e6auZ3zivLuc/qpiPGlCiMekZAWzqo6oj4P4GINrMozEe1Ae0Z126ZqnjmYZb0JTrcL61tW6vMZ6ZodXUKtF0+5cXWI94/idHTYXP12mf6ecs8SUPEyivbop53BbJRvMK5ld6q7HVDFb33SfgzmGz+VmHdZdHjhM5Rwcfrt9QDQvf/hBVT8w0Sv3Ucg+9fzO9JH/3OCp9CVChwdw6qLr8syT5ROIoaLlG/vsH/EY5u9ShV13nmVuERvIXpuP0hv0tPzDaX24etQ/DziLYUvAAuPxKc7iXr76ux6wru0tJA1ROk+wyDwab6++SqB3ByRueJw7kNs8oDa9zNPMZv4llId58TpaigKjzLG17mVDJ7qmo+LdlZC8xVl1zWnEyTzqmEyz1wY+JjsBlvc+Fqfr1lCxqCVXpHX/DZWonlMjuC6vcvGpTAXqC8fEuKdE0J7TDTPaGNJZzIXyurzFxMYHaVMmYMfaNwwuGTHxADqAJiwOHeetrCLExkKJlV6k4QmtQGVZ0iOouAPEoahXC68JaZ/4avtGxBWpdaW36/wA7LeszIYysAvUO7+ApxKdIF3/wkv8AgYzFMMG2CyFY5wKjaMH8G0KQh/5bK/klJUp/FSkpKSv63UNf2f/aAAwDAQACAAMAAAAQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAATFtIA2f7NJSA7flbkslKAAAAAAAGo2rg0Ummy0v8AgEgiAn4AAAAAAAYFeu0aitgkEAQWMWJWfFEAAAAAApfKkQmwgi0EiSWzOTsA1h8AgAAAAsT3KUI0GkpwMdtfALCQcv1kAAAADW2nuQmAEFcDbZu43OwjNeRgAAAAAczvCq70xog3/FZAA727/C4AAAAAA/Jsj9oHJhud9RMQD4GgMQAAAAAAUf11PPSjCiIESiGFSgB8SAAAAAABq+qEgyx56Us4AEBkkFvJ4AAAAAAeu+YtQyk0Ie5vtQwxEfbfAAAAAAD7QYlA0j7EKZWje0BsiddoAAAAAAbTrMSNXltsLfw6mREApn5AAAAAAQ4kV4CoeTXZocTJINahxg4AAAAAAHPlXtAjQ31x27ttukNJfhAAAAAAABY7PolVSk8ALL9neFUxCIAAAAAAAZcgAelNobHjtd1ZpAy5IAAAAAAAEDZLOx2rRF4O7j3+AvJN4AAAAAAAarsqIfO70+KSrTmzKprAkAAAAAAGIXhOwJoLS6QW2nzotMafAAAAAAAfzWBv8CUXVyYkECKjagvRAAAAAAC50HXYeTMOpJAautSL27vQAAAAAAt+0hIUsMLf5tRFD7S1XuWAAAAAABoUlvhO82tlMDxhrI3PT2oAAAAAAVRVhzCTch8NgpJ2Yt3xI2AAAAAACF7XxMAnHNQg3tutrY9CHIAAAAAAAMtjN+mfD3S6++3cMWbRPtWAAAAAF+9x32yRbsNoc+hCS+AgO1gAAAAAxlkRJ7HTjHF9cBot1+iuFl4AAAAAJoo4vzlftuy4plvOe3EYggoAAAEkNslCq/KPHyCQNSXnfard5UAAAAggpIMA2sEx2G2BzT1WY5fQqQAAAAEgkJxmEg1iJSJE6whOCxG/FAAAAAAEAsVlJhzVqVImOyFzxiWdAAAAAAAAbLWyzGNgBtGQkt6MiLU0AAAAAAAH7CxitAgAhsEAAgNXCGzgAAAAAAAdlzU6KHAA0kmk3MkLLzEAAAAAAABLeAZso8kiAmUEYqPzAoAAAAAAAAkAEEtkIEgAEgkg2yEAEAAAAAAAAAAkgFlgkkgAAAEEgAAAAA//8QAKxEBAQEAAQMCBQQDAQEBAAAAAQARIRAxQVFhIDBxkbFAgaHB0fDx4VBg/9oACAEDAQE/EP8A9gCeLHNSXHm7nFtj62HxYWFhYWFhY2Pw4WFhYdMbCw651xs/Q5Xq7+cthkW7qf5fm7Yfw/zc0/7G/i2KezecJ/X2AEYAAU/QGOD1/LIa5qHfVZMZagO/SzrH/YU6kh1GbPzQoGPUZs9fNmwLm+eS8Ljn8zFi9PitCcHJ7f7kG3s+sgyRWTqDD11a+E/E/Hywxsfh+CZbVnzl68kYh2AHLy8v24z7w25/H2torNaX3f8ANkv1id3UfkDHX30aWlpadPfrjY/Dplhj1PzFAbhaQMmlQ+8Rnxz9rGPCn2u/ynIJJubNb2ZNEd56ynxGKd9kF18A/GQ51O8FjY2FhEDrpafCuqYr1I9zwfiFzZUZPPhmU+Vkw+ZmJ4/raQ+kuW7tzn1+AXbJ7x3+AMd/hO0ccRyzoMmyfDjY9TyXIveHL7E+EovrG15Jdd+bkznZ9+IfVQvPoPxe93dGfMJ7x36qI7/Fu5drgoem7fh1tevbci9/6hv0CW8SMeiyzy/HPzgP4T8x+gCPDPl0knG9G7/i7uOp3T2n7GJUIQZMWCVS1jhu834TiXwnmGL3h9glzaG2VPbf6+dhaWie5x9nLQYc7T/R3uQx8gLwh6XddkXmwP5j/aLJTWJF1jqA3ZB57/CwfCOVzj3hw+l3SPZ30CH9/O3OCTS9xh9f9bkN3J5/o7w5EdAMjt8HG5wPeTHeRhzxtOYiBZWzkfr4mz3gBJ0aOQ9ZDAQQU507R10u50DbW8so5MuOFr1fLcT9bsSjifgfxP5Av44+cHK22ux/H/bkWajpsnVPh9cLNyt++WBMXAMLEe83CnRqypcFyahXsQEeVr6zZAnGxybPnHrq17Xf6WNj1fKfL9ZYOlPtgL39X5y4rQ3Oc+//ALcX8kEcTrd7B90g9QWeNbPLduYRinj9bEtsatlgIVjuXcXmu+Xt4sRaeICN8kz1xmyT8etaGdntMg0ba2PPUgWWdGz2Q+vRzVhL5wfdyXz84NkH8I/zfQ8/ElRw9W9bojbw3UWgWXitpYjvDE/fvcDRAk2DcCRzo+fpeFfCMvyc3FHMiBcPLbAcxjqJN+5ntl3qCQcc3R23CHrQwaSxzbfvdIbHFfLf2u+Hz1iWxOoZ/UI982a8frFR77a4V3xkMHXUPc2gRejggByF54KdBj7/ANdrSZ5ja0Zn82PR5689mz1tR9LfzuwDub+7YehJUcnayRdvR7+oYOvMN47RZyvfm8I1PSf5/wCQ01/QK893/smCe9S3Xxv9QHJbwskDUlzrzNC4JjYbe4sX97ZjgnIni7hbMiYve1HsXYXIjVR47zYD95BfAQ5769A+AO4zx4J7TtKM4V/F4WStpbpofxbwP0Acn/mejZeMdj5PX+oq9S7+zFyAeLPjjshz6rv0eF2kcaZTqjC4YLW2OzC7hviK/YZwecLmkuTMuLgrTiNxynMwJ6XWLWy04nPpeELinbH4oRdONuCZ/wAYbj7QIeB/F70K/wA2G/oNHcbl7oBBpPNWf1bVux+7siRx/wCbO9mc/c/zL1sd4IMHe0xfZby4Sx5S/PJsVVmwPOwl2z3LaGexZgaeuxwdrHQdpg8Qs1CahpZznSPUJ3fSI54/r/F2tkVXdxRMwxyeqD89m9OR/mML0iFy72cPZehHbmT3LdTvDeGR0GsBTPaAlnvdz+ZZ3S5JgnQO2EAINAwjnimkHd+hrEF6o33WyXvOvHAGlocRg8xlwpDtAXvEks4fXi5g9D8Xvk/Dv6OapbhGf32a3nJJj4tmME1hy4QOi9mfGQWBLaHF2p0sP2MuVN26jmTN/Mi3Mi7LivMAvTMVF9izQHFjroq4GIcxYXwnU5E3cP2u1/k/MaD3D8WbPG/5sssfnmZM4/8AVxh32c4S7C9EAzOmBuZLgSgHRbU1uHJGt6RQgOYKvYXGok+5+Yas2I3e95bD3n74MDcuAxoPTAngkeCRsvc6BatHaZ3wJ+tR6d7cimWh84cUb24fx/2+1P8AUAiSy1bEje6c8oL6xLtnvZ7yXaPJZrqJQY3Fa16SDu2n5gAj1gDz5URvUw8NzYo4gcmDzBJl6WGAObn4MgbbOn53qjbcNz7/APuSJ+o/1Z4SeJIhMYIdEA85BiZ023bYWc7l2tEGxpa4PT8x4/d0yjSko+Vd/U/S9y92PXj1buGR0YTOdwU1QS1vzW7icDwj9mF/PL+pNxOg6RL0+e8SyuaZztaEZ6Frcppcss9cjXhy7P3dIeNqNkSGRmXdm5fFtVXr5DexANHyuVwYICtKd2/PHHbnrsT+Ql6xjZgQLXTBb9I1pd7PE+op4PW1EJLCX4JPKlj98nCWm536XMjzYTT+l9WJbYswjAjTmBmHvNxLFIc39A2x/wCrg+jjCiHIy4GxnQG7zcZejGNgDz0hnrwj8DeY88++XC7E+L0sUdI4uSe3RBTTqOiU0iSOdo5WFsT6w5P0HK+d4rXY7m3qL3rPmG8edJauhObjEWGc0xqScZkvF2S5d4ja4pdEc+c5YMPVvLYaLkDpDKEM9AhPc/NiZ6TxnklyckMOHebbfnNHrYt/u2KsTr3ALurgS5yPmOYSGQfMzXBmXLCjymR4lAuwCBoNt/JIGt2gWILbbkOPXQZTZ83InhPY99Cxt2UvX5mVscYVq10a4MO88Oq7V3pHAW8sR5o6/aexb+ZlDDJKRnUBzvAM6Vmi1cGWCWU7w6C4bJywnEed6CH6p3nZb8vO0gx08yxtx0LDqGMpyWyuE4LYnu8Sp9M/Npy2FGKcYjN4m7JlcScrz2IszpD5XYwg2COIQgEg+uYr2WzI+WGKQ6S1i8z1B36VA99lnbsjBUqb2jwe7O9k4/J8IL0CO136adtcuZMT9Y5BISuilOGFNuy3e8iB4hyvul7fK06tYhdnp5vHXzm7IAcpt6wXuFBqHJPoTknnmfn4Tjc5M6d9ouSDIoljn1ljHS3uG9MX1XGDtH3TDkT7BVYIfnAxO+HHF5th6eNmXsIbJuloXPREn8IyHENS9PzK2vVIOuVi9mT2saw/e6Vlgzvq59dYOZwbx3sTxWbi7n6Duk0jnFtD4C9Ba+zNNy8unfxc2E8L4jDMzPNiOyx6rD1OIdizNmEt1Oyw9pEwIL3LzEBYaEcN5h2gwe9oLLf0ERHYXLc0jJSz4rfRFm3/AKDKaJeHZvuDtZMlwfAbYWHx+xv5hKOkj4taW0mBeS31gIKx48xe8+XTBOd/QCDi4u2Z2Mdue7L1Wu94Z6f9MYgyIAcTBObM00ZvwNLT4zov5BAORJLmYmxL5XvadO8vToxq8ssI5FqOeP0HPoeGypO+JQN2qWBC36r9xjhYmd6FiJa+fJD4HhZlhg3iZ7xMRJWwfNxcxyjcIPP9F36N2C7TjNhcozrvJnuwM4MIux5Vt3+aHJDG0FtLXLbPC0tJcRzlL2fpAhlD69Rxiup5/RB/+E9+g7/r/wD/xAAnEQEBAQADAAEEAgEFAQAAAAABABEQITFBIDBAUWFxsWCBkcHwof/aAAgBAgEBPxD/AFh51fxvGShBfp0tLS0tLT6Vh5FhtLTnS0+w38DMnq6jepnpJft8xC4j935gbnmYWn3d+3E4/ATjAbJJl9Tjxh/MvdHLfqAOVwfpVpxDwIYOTouy98N25fgiSYpPUTI9YQ3bfLTg2TufpGWNsk/cfmNts9x1YWEbrHAdXrnbThttLfuphvcxfTvjAx6JXtP2xjsnCzBBw3hhQ4dvo5z1lyHC9WLFnCd3oyCDwwHuZQyHQft4e+HhhYBuXVfMURWSXjIYf3atciu7jLCwinPksTY3Id2try2CklhBCRHh93uLor93u7kd5iJlhtbo8PUpOHeOFy7WR5Y2MfU+Q2lpPkO8+7Us1h593djjv3wWEu5e+IzlxPlyevJ5MIJpxZMDvge4OR8jl8vWffBZrr73gtEvF6n1brwpTw2EOPmJcFKeJQ/jF8Y+pnOIxHsEseHrn4Zd71ysG/e9sVeL1dCXcEls8jLO8xyqx7JAn6DM7VoWSKY4DCXzD88nzYMvfIzH3j2868XRn2vMEcs74Cw4jTpwyeK5KU9yHyu+IWJAQfGGzbO4MvOT3SPo32feXfAsBHE+8vWwsHq6N6T6kyWFb9IQdDkYkYYgDb4yREexpiBY8Azj9AxMfSD1DHe42BADD72bN+XYl3su0sXAcGxSfZ0TdNfK+9UPWfB/l8iSS+QgdX93IfYtHe23vy0kltt5Xbel3bzbhg8PuvANS1b2WNWGM/GLjanwI9LsRhNn6neydHxfpBa8rlaUAW7EyYcsHPaxxFJ1LemXc48u8s++fLZnUfUgjbfpxpF6p0nCdugtdLJHV+PnR3sFI9S3l28Ego95E++D1PVu8HQd8Haw/gZl4BXcGSzZtPo1vEOmZcFoxviL1woG+t6zeg4/w3vesBqTvgOZ6vlcdXand0Q/hAHwwN6+hEoUY04+7cZeqSUfZV+8SIcGKdQttvVvNIu2wt7LeNiKD19s544FnxeE+yoLvGRFrkN0LYMwS/A3Y6ZM5xnEJMITx7Jtg9QYnV2OrJoCUFklnywe4Pue6yz48EKseOQyFDdroR22Wh8TRcNl3xZ3BN/A1EtKBlAdiA+S2pXOtn2aLd8Eh0RmXbwadZeCN9TdWJ7z6MHSfr4jJ8N2rrwGwvZlEbkODX8L3XI1bMbthMyY5g/PNjPuXpcJoWs0WSJky7EOTN0b1SbhERMkLLEGOfJSsTuH9jmz1R6ulpY++Hu3j0T45zFJUyBCrM/QutmW9zHG7NpwZUg+LAP44DE/usB8bTCxOMII6t0WoT+xmJ/dP7I4RlfePIGv8KfH0jPiE7u48jYLUplhDrz5jwoJDTT9cFtV00e93s+eAdxFDINgmKTZJDudCX8k5yn975tYhgLdDZ9HhrJ4sGFgNtJyHAdyJdwmpOacd/6Ydwdrt74Zspt6E3bh2Mqy4JJGWTKVl1lsi+/8VkH/AL5nupjzFtJsbN22lvPnPLlA0ZajIDs9h/8AC/4l3MGGcixH8LXbAyxIswBtv+UmYmdG9+RNlPT7xILtP3lgSYwhFSUi4A8gx1lvBSyPYPUTQ2k4uXZ3/wAT/i9P+9o8y9yxtWNixHKxhI7jv2hAwbb794m9n7ujLXgcC3qBe27wYOhsxlvvFSyLZhFo3iJYyUXf+n/DLFu36IHDCNhRzsOLgMR7by0eJH4AWkFfu1+4Vv8AfAPALDifCLKbx0Ee7KqGCOqGL+n/AAy0vGBvrdcTyS8stJGOaQ0/UvdfNL4turq21Hpy8vfuaG0d8pHCQeL5EQnGJ2nWxDZso9ZKReIs9W9/T/iOgQQQdyes3ZBdwOseZZMjysfIbEg8xNOAtETtrFf0fcS0/SHVB3x3jwcJ+ngYZkQWpMkxey8fMu1LRpwDkQ3f2J7fySPzaFsad4MmwgJg0eQ+Ht4q3Qy6dWMex5HTn4BdXq8R3DiophPYeiMJGi6L/wB5YDuZsG5YDS7eRESQZ6W72xdsH5uhZpKRk6t4EZOeXRPbv+2S93wBkwrvjn3fwPF88H6EJhxGQhdtJ7u4nGOljebYt2yQgfuRaY/iCfV4CF3sSw5PZdG66e0gdSjMQNl1PwPF8/Qe+A9tbsEHRx/YbsJcB1LGU5iQPJ29k+JT88WAj+sj9bLpM94Hac3wgJTevUm3eEAYnBl2D+B44ZiSibjmFAyTyUVlGBkIut3l1Pb9GexCue8di28W2P427RIO5vhonezO+XUXnDwzuemWtoLz/A88dDbVlhC2J2XzZdDixwhwPB+wJhZTnk8Xd+nA2HzPOro2eh/XDtGUgu9Lpu9rueU9mQNspdRDmkcAh+BeL1LVKG6YMaRAio5VikiHuH/UPaUEhrb9nzOQhnHjYTw83cS/4bX95k1iao6BTPV8TyfbFbd4J9cdfgHcivTLfwpF6FhWdvyQ/wBkf9Xas1ZPCDNPqCakJcvou4fxDP7meyJAy+WF03XXHvMulN3JHndnfwFliS/u3dk/zT9huC+NyT9KEqluslskP0rlNal1u98hbf8ACGE/fHHy1iHMvjLWKjrYMPxBc4DQle5C62mM4WCRQS36ekrgYcl+heORDEURF3OtqdI/b8xTOFJPvJbt27f+gf/EACoQAQEAAgICAgIBBAMBAQEAAAERACExQVFhcYGRobEQMMHwQNHh8VAg/9oACAEBAAE/EP8A9y//AKi9APlziEb4cQGmYYVil8ZdyE31z/QblfH7yvj9/wBCsn+lPJk58M+GGP8AWcWf0nJyf6zk4nJycn/hbGdCFFsVAg6xmabv31OKFXeA+azT9NKl9PnnW6ALbx/jcir3oQfjzl5iAxDZFdn5MTfIkwteb+M6zecVR73D6uh4wF3PUkOm0uQbqKvTiicw2ZK/lgSKzWWems79Lk8/4Y+wVtZz9MXHYEu+PeT2YoU3HPXi7rtNL/Rjaxo8bI/TkrUqTm5h7oKLOG/R/GCcllw7zDzrtH55hXOsa5glI2dfLHH6+JcpmOgTheeMbKr5S9/libRU1pw5ogExzM7Y0UGm2eYOHXQNed4wrpLBq5OG+s2ilZYcoDcZwd00mFk1XAkfguWtDpSJ5MCMNUE6UrKcT++8ZYgApygvlvBt5vnihpz1iwb82wRWc8k85q1oVup4aavw3g1u8R+haZR1FTfkD6BlPIIUIo5r8GvWcKN6mEb5RgOfXrAH+azf/DDY0c+J/wC2WIgjQ457yDzVbNJ94FQrk0oOPGN5rLJUgK/nLjTQq5bHkUMFA9qLgd1rVoJ3iWUv0oD0iTNrsDXiuvrJeueIZCuzfCaH4c6JgRRV4bzlX7hFKOh5wQUcLtHdf7zkpG7gCSaORt4aCcTo/OME25OgV/k/OCW7W0oeNPh/DigsNfL4DWXPRX/QRfhykTESvmO/jK7pLg4S3WzCbzQQE8rR8c5frOlByHG/jNmjw6JWA1oEUCOUPeC79sPLmecFhVtgFk4vL4zrfVuBBxrgxW7NyhKzvwf0K/u83xgHykHnd/C5rWHZf/MVsa0QD95viOkPATX1M4p+jPlsPpZ/Ts4cAfOETm2HoRgp5t4Fy8oThn+cBsqKsfcmvjHqgKB7GNvNyexCHke/nJw1WnGudTWbvFQ9BbnHJRyzxFOOOctYqRq0YfbnDhGFVJs9iLjotT2cNWeIQC4oweewCPHMtqm1xKBaxnpWR9ZXsFAGV0ajq3IhUAO5DFxVM6aSc+HJpIZ2xiPXXjvPJ3sII34be3jHEtEBcTXxqZXyvSuMJCGjeOXtc0aSx4OcG+UiNRO5o4Ot4b66F3z5ZYshpIodAvpiQjs1CEpb6zH31AL+Si8ARR4+VDJt9e8XUtE1JFGcjxcTEoGYZemzXvOHrPxelgvjNkC6M9RaXjLbpcsnMhpeMhARQrbMMCYIoAncZSZRJx24C6H5xnTcvLcaLl5d/wBl2OStO4Vsbbo8Rz9UniQEuAPekiYIqw8Gk84OIAEemXFPZgPGNEp+eDCEDG19qYl8Vvp1cL8zRm+JDoRW0buj1l4J9iPnXjxmm8zRI3ahH5O82YFfI8FrQYYGy2dau3y8O8M3aI3QSsI9SYynyYWoAYIpJs1zmoaMpsc8KI78YC1DLZQgeEK+WsDTInQgSi1R0Cw1TXzieqhIyXlYVREJqrT/AKneTMJ1Q5wVwXfWGkMEB3E1rUI0ecPVXURpbKaePDhqbWZHnfeW3yMdK8HHv+MWdpAKW/jCWER0K21pyc+MECgqMl9v1h4SOXlI3sY35yHLUwKurdFw8WSFNQ8tT/Th8HZIBa4QpfgzmFRGRiimpy9ZAmAdkUrmdYEZ/VJbGDcG624iwMbWh5f9sbUKju61PDr24TUbiQqRsyis00SFdvYwVywRvJQsd6byIVlOnKXVKcsXhDI2viEHznjsB+fDAH5kMfbGEXdMn7QxcVj1W36qfWHB/Z4xl4PWafT1ima7jTe19Y1KoICZMD875wZBNlfHGIkvayhCM3+GPL8UgbT5lPxh4EaIh73ywi+d0hxuIbc4hgWRSbM2jfWdDPpHnfORpSqhCaKYJ8DkSkR5Dv3ioqemTWjR5wGZrbUUAmnaYwqii15pVGwvRxKj0kAG0OQ7quW1kBh2+cDnQqmk4r0+YYP0Da/jRbd7Ppg4Gg0LzAqOUHaCAmzlgLzDjWlcgQnuMXZOuiukjp4o9JgvdKfEF0LppDGwxHWm9nIR8D3jvLyEgcSsrcAJsdB6/vDpWJQdh5OOsAEd0OBHHhTIryyhXPXeE9ZxbPTNuAZGnbiZQNYsq1XXvF8SUQR+MESAY4c1XFpYAiHjiezHV59/BoUw3+vdimgxdYkKgnNhr5zaWDk3gTu/OUKSYcoP0uEhAUOgIZ2f2p/RlhRRyH6NMcM1S5NGj5MsngClthAxFUTmT06DhDoaFPhHNJgfk5brh/GLGBBmnMFMUxhTAPKmTY4V3IoCJwP4x/JVjaiQSacb3TLpbBTaReZ0DwYVlLqQkK6oLsbWF3YKA68ARlN+8R5z9zwhgpvJLjkeGD4xr3oDQ+C/njCfCITINC6N54/XbGCBqATYcxyZNIAkED9hnjB5VXLqA7XugqEx14VEwDFFQAKqZoD6INbkq1CF2sBP2G9Gas7TNmHRz0eDAEsVQHN9UA4bMy4DxFJmvKoe0xco4RBVRNO8Go7imTp7zVlW1GdKaHXkzdlXYW/HBBhMqFVEJ0PGTd8sY8p8TgdISFKq4w09uHQTgxwRq8nOAJmkXQCiu/e7hVmaZDhhfbLHi7B6wKNLgZJ1rCEBBzBo+r/XCMwPD+5GkKwqMQ90Zv3iUxSN+xzgBOeBc5HOhKEzVrs4jikBTKw25ckbEuLkJJO26PWeHQHyNkiU9hd5dpsw8Cqk6NzjFNRvT3/WaEWab59QNug0ZVH4qUNoonQomsMaMOMJRDovOARWwMh+YJtY201lInnQF2jAsMVYAFESnIeck1ojfO2T8CGJnfPqLA90fTgiTOdZ3NvwkEuXHendrio0DJcJk4mmEKvybzegwiWEq4tLO+gcQFbX4ti12vNlMhdWexITo0ANZdu9Zjab0AHA6hCzPpzsmDpsTJyVH/QnnN6f03DmwzZ+M5kxxajveDPjjsoO4dH1njRyW/RwoJltWQrZBD5xDKqpCl79nGKV73QO3HQHV3ihaC1e85xOFqgAoNNvjAH1VUCin5Yx7SfeAAX5c1fDhy4xKdwn01/vUuRnYD/OBxaicBjFB84ljq2n+crzKMgFlg8G8shwUhuBu9MMVFQoo/Exou6HUVe6XmtxqogieJhybiZcIA1PWWwlKZZEDdh6JlgPpEk5T2d94e2g4T4SGKKZqhVZlqlop2YnIpqzBhAaTyEOK1n6TQrYF8Ye0rAWzsfDG/WB+XYYiKW8un2b5wjNgIKcxXhQmHZCwA1co2R4MhCYpctGCcOUjaDbzA5IpXDYwrFmk+egS51TCAXGZHOgFF5QPzvFfnCzetJd7AAmNfTX7MHmJNqG4GwZsJDBc4uQ6KabHG94067wFdqQDe8G+ZBRl5CE3xkXZAUGvbgL4yklrp+MXonIq8zjszjhVk8ATvBVran8yoqh2POMwCLCBp3Tsd4smakJuNQMP/5hgX+PzgOsNZzzQ/cql/0n97cOzDpvwIaK/Jm86uL/AFeVw3Yyd/5lgw4VuAxC1qIae2cUxqLPgZCQjSy8EzmHUpKtNutYq+3g5f8AyxBmBHldB1gKMwKhqOQP0Awy0Llhqx1iZV0EZBBaXe3DMHR+1kol41pxMZpCHk47bB48mABf0TetFHoci5Fut1ve2bd8L29chJeVaxthQqcDOmGc6jnXPDk4DNVd5K1ExtwjtC/GG6E+EQXcIu3biU+TlyiGwCXnczW9n9orRVjVW6wZqWAICNhYT0zBjD0gIKvQPxixdwQD94kYNRlYg2g1ADecJAr9wgEAEo0ZUn0Xxp6ZrVgAsYEqBpescCyGS5I2aJzrFA8EizKS6Y8BQC3k3g7+LgnJICuCLAp7ykWmgpFUeNZQ3dBh4D4PGWTGDpCZkQD9R/ddGcXN1q07mfen6y98YnvdHjlMbJecAcrMM9kigSrxMElsjZ4OlwUwRx3ergASEzUTXsyneBRJxc7eNqmInTTUDBDyTjBJYJWN9dbmGiwVp8g0CKBYOQhVJEQAF3U1j5kiqRiSco75cYEA9COBMpEdysJir3Eh6+lkfOA4IJhHAKXDo+gXdBNartxjhxUeqKdY6E45xe3WrZwhsBS5isxISxWqEHt4GRsBCHAntenjIycx3lMYvfGxxHJOJ3UZHP7XvIRcYAujG04z/ppGy+y5CwSQ2mSGzl37xaQyW013Tpx5cfjNiiVAO6HaawwxJ+2AhcyBqDFTwhGKjgVK6HLAvblSdPL85abAWppHw74uDRCdUhYL5rt8TCfGAgezUHlptyOnRpTaHe3PjCtSJpvTvrHJ3jS2x95DptEd44IDWv8A0L/el15zftvm0/fR/hxcC+4dD4wVWgzQ5EG8PziQySg8+zOFDQEUj/rDspovTzuTX5Ofyw2FjOfwPnAC64FCX2/jJKl4EfYDL79Zd0oQJaC7sB0NZpSwWFYAJ1ki93DqlUyDgQIU4TCUpOkBaNKBryPeCOntdwgh2N5HvFbAYSKtgbR2RbicG36EWjk9Gw04vmYLOjg0l78YcLnIeFDl2B8mH9BsNQe6NDWh1hphFAeyx4E04i/MNHmGOKIIDTlnTUjFMhzNZalG+jQeQUAi0IYKlRYIQaDd+8Adsj2UNc4gxlwvFH45xRgCgU21zT94/r7g1IKW1KetZON/0CCPgJ494KnEQtB0n/k85FpJSdCF4NN+sBCMCPwsvt39RFNOI784AtzIftAsjJo84ChdxSlTR003jE5SQNNuPrFD9GsZ34MMlkBusCfA/nAB0AnQEP7xxj3vOcms/SPvPYSR84BFsVe8SkdowW8TWAT08IORdF45y2WgmQay28JgQSwINY7zd2cImu+t/ixc0vTRxRsd6YA2RFQ5Aqhpsm2B3MKWWFdSFqYchosVFFcl9sAho2zaptKw8HOFwBqaYEb76D8sel2pSLZOwQZQpgLjBA0rKVuhjiJi4wfTbuf8GOtqRAg6+FYJcINrG58l0+WhIGomMQNwCW4HEaMINVBMbh+RLATscdmBqB5SihlAJasc0yKmnRNgDHm7jg6I7AcDlpsESd5HLUMgdh3NMGw1gpCZU7B0ab8YeYRJ3P1HzlVdJ0ADHZumA6N8a0ygVUHKtczZygfJRv8ARjCgfWHlSkDQAE5AHwxt9hE4zTHAdDvCk5QW3VkpxwDYBXgejJreizmbnlMV8dze3XBo+P71mB3Z40uH4cHeIexR/lgchCTxhhMARPnU8GTkN3z5rp1fnE/8fffAEjokcszA63K3aSBr4HAxqFDKtItaev6TIsvNq2An6c4qcbtN34ot6FcQ6my47srlsoKuWKA1BNcJ1TseMSyEG1iQHn3enAXugsbdoA2SlmsR6k5IQddEW0B4y4J27bptzusQ4lGtOEUZKaU0jRaYVVYwHdF0NKpp1GQNwdLowCXC8KIz7QE76ZOTgD0B7YbXcPIxDKB9G3B0H2ZNl5XVCe/cTBZuURdtt6nff3con2NgWyJsR7zjngfa9VJE51CjnBwMNQ5/8wbModOjxwRa1UhzA4R+SLLB58dJpk1wvLO2zfvJ7xTph6zZi/KrgjsEmqeMCSodRGcgqdhtconPhwheFZU68dsY9UCe2Ph4scGn9202Zozkx+NQ2lmeq/fD2FGFSg/Ssqjg4HAYnn/EoSla4gibHYDr5YGejn/45R6p/qYe8kdOvjFgXDSHdCmh9JgcYyl+CKrlRhIKWLPMC49AcuvZyzYDRyzCRUSZEHOMBpJBELgSQ4V3zuA3w6esDWcQ6QGaANNOCFqebZroWsL0Y+Xg0RgBEW7IOnn7FCiFsACH56ySkhOnkHzch91DhAnqj94ql3jmk/AX7x626UEt5x4JM0695OrZju4187sFJasc3byT1puUi5+sMAsd8aE75JGfbPOQfWFiDoh5C0Au+Vzd7/8ATEmlIp8Zb/F/jCoJrQUvMjjh8scdhmLImhrecrSAiBoeHbgSZpPJt/bjrd88pnqjI8Y7ceRBrfZ/Jw2f8AM4kxPbeZvrKZusHnNt0AvzYvC/cyUh9s+Cia9MpnOim0dq1CxTBKxOJNCEGRRLSTjGUVmoMOiAnlfGL9v+WF7/AAE5hNCqffYhwWQADILF1WafGV/1YitjZATHAWEHNcmyUXbeXp4JtHnPEZ4HrII0TYlTGWLHZ7clWmxKGqYqMkQO+m0FwaYH6sBnaiACI2mWCBAKp/rvN7KrpPqhg6O3OaYdgNkNno1rDXIKd0PvAHHKNuXSutpYwPXt4cYFlUE/zkvSY2PL9gZMPinXQraBKJbrGsIqBVuhy7xFfNIusF2gY4xWWt8vnNr3xt5L8H+MlLxzqQ2ZAEieGOm4QaDNAzWR5Ok9WPdElw2BUolCOhIDZB3vBAcubM70MCnbRkQhf9UwbZwP74rmed47OwWji4uSusW2goUx2hC+WSuiAbBrDJHrwtHUs8RlxMhrvA4gmRNDY07Mll/qsgUUGzcyY2n1YKbLcvGDdJtzIIeym+24CVXgIBMjSO9sAhYOiin1N5wtf0IFHfWBIXITyA2taGsdH4zCRkNIujnHkBiy+jhpUdKOClQVe5O8AeWJC9D1Me9Yy6dWv1k0AVmD8+MfhdR2c3eA8JN+ME1haqqvH4yxqQAfbzkSsY8NhatiEDBGu7XVJG3pA55X9Jq3Ow3udXV1GNE5AQUYMxJngx4tA3K/f/gzeOGwml3pq968YekTBO4zdsQ7aHC4g3faWgeTF2YHY40A6OoB4F7/AKJho2OBVUJPNjBZBiSAP8ZT5f3+MnksUrAZ+eIOxJgaCVNfqvBjqmbwD5PGOdYtggvVAfKawSYLHdSBHmapjBnFVVs8qbMDIfbF4Q94/dlE4ntlULt3Rgp1XZoS6GhXoMIaLAuNp4HHeIj7SAYBtAibhcGQHVMlf5HqbXFXN8RlaSAexml5b1ifF2YzK7ZW1AbMJ6bpG0W38J+cZlSEaan8mIG9IAAPtxvopEmH26oDhe2kUOTFQXJlq4NAqhK2FKtF67JqilNMX1ImEARA5Kxc1Gc9Y1FO4nQ2nOmkXKSjnIiNs0gATTgWeOk8arkA4HGO2P7HgEFS6Qo0YlVE6elwERgZFyrtt3dMpfS9G6bzT11h6vAdKEYNO8c4oYIzxD5GZO5uBiKRghrcPORGunsAEA0ibq6wtkrsEdvAe6OcEwUdC2bBvzH+WFDkDFh/wAiAZLp1/OE9SbSPE/Bmp1C2swEWCt1KZuUJC4TS4AJXBMl5dsgg7DmkJcu782CxjWwNUA5zsWtGwAaYbAwdYEHWyQJiALXeVzGmeLLSKWdcCCH2kwjIlIUKuD8oTDZbjtG+cqwSG0NWkZNoRmDkaYAdUZbIYHZk3l4daJjDxA4K2yrEvAeQC7Jpi6MNnSDJzEB7YOpORA2Jh4R7DJGYnDT06w9m8QMglWmrJG+o0LjXmsiVznUvOCY3PukRnKpHmQOSBbsRCqSjVdq4U1yqMG4Fw0SOVLkSmhV94n+b6XFipBW2na5XVNjjY4rYejE9wUIx5Ib1cTSowo0L8MNZdvReCgZRY4H/AGV1K6UAsoAI4EaeaCDUaQbUeeL4PB9hQg+rjQ6LI0Rq7zyGnjH9SZ5+Se/ao9b0e0jzSMRhQWHOUObUnxzU0ELK+7+ZgQ/oNxA/3qq7M08HQQP5BhTC6gIh6CpbocLyzPni6CKyiHON7CFIbtajbrFly875AQUDMl1jBJOAhQOLMqUqBIMTKIwEW55RxTE6jAIC0qOIoqBsK1LK8k3Mgcbg40NnOHkqSFYzhTwHGX0Hvn5gwYJKYyAcAXhZ1gvvu0f8pqi8hMDcpdoVuyrskFXDP4p3A23QA2YYDJjQ0f1gY0VTOwvGqoKnOCnQzUcB2aZpUI1CkMT2JkoUCeFoELhAQgCBLYFxyDNEl7JsQEHA6Dlv7sf3RYAisHPrKJIbMbzyzbsEbWh1RvOYUR5OFWlP1ixo8FvDjGyM27gAFFtxUh0WnZGkFdHZSOIsv5nrOjIUUwMAK4HTq0pweMNCEnPQLoWOLjvBNmFOUkHw7xqsgMJdobiwjvPRSr56xt7VTk9j3+8OCDrNvGIfWMZ7/vNPeIyHKcp35/Ux8I1BHXjUC8y5ZbpNinx1gdN+bkMQaAYPiTOyd8cz56yVVUL9UzSd+dneMuaVENhQ04by4LOkAFCzs5XtzRICXlNMQjCa7zn8Y0AeCysk2v8A9Y/pVrgw8EdDml1XHrmxRZxCBbDVDARMCIa6E0QM8HLHiWAlKE4R9SRvGy/CZQAGCB92XZak3SnK+Pj+gRm0bI7YPx/6xyhEu5eaE3apOHDiIQCcJf3OsPC3U0LscM3wIOHjCbLsJMoPSk6kf3kimEM5RA26jfz8Y/8A23QGiAd0QhremB1BalFVDSlVV7DrAwSe3EX8HDnAwd8QP7cmCs7N88YZeQ7bTUbdx8eMjE9CT453hrf8dqt16xJQuz/wMS8S+18YNUUFHdXjChEQgvP943+d5tQH9qrfj9nFoTZ4esyfbNBcAOcjyxCFfmYO69YpzZH4yKNVwbjgL1zk7O4ZTBfwegYf1B0MOA/uf+ZcBAvohKuLC4HIhbRHRDhywaqqwHOO37Ys4xyMBQ5S5Gm8R2/LxnDcWf4cogDTgaJyvJ/ogyMdZm98OM24QneIIdbWQYleN4b6TnGtpXzhpv3lnvPAwNtZkhEnioPzmj/UKNPqZbrG4I487FqfOCQgR1amKlKCbyb5gmjemB7xTBIYnAfy5+/7wn4xtcyzdJPcR95Cq0rh9lzlzm/WQ5P4xxnn94ziu7Gl+PvIIr529JhOMtwp+cLoql0Y+FOuSYypS856P2YBhcP9R5fWOysPee24OsR8MI0tfwM8IY5apCWo5RSnljlL04AFdv8A4McagoRhJ9x7Lg6mBNegZHaMEnD205Vo2pvD1XlK3ziNq/XAKewZZz25xoL8B0CqvrBu4InWQCLuYD8nr8FSZt/JjguRHdM4xGess+nPvFvRznMGILFaSSY+zFkfAuxq941LmDxQWHfGscITVjiEj+81IRzddHhcHvAb4A/x/eabhlUeNLhPhyqWDeCv6OCTG3g/eK4ftidSHHnPKPvGkkem8fprICxs7HZjxBDDyLHpD6yuvXZXQzaVyjZ0XB1CDuNAv7GX3qDzhKPortkuiaPJi9aHYpWfb/WWsGTLykTrxklUxFyYwjEAVv5WvTWs3p5JKBVDeUj7woXYgUeu2JsaDGEv6BBob0p7TQ+i/hgoJErx+T94n1oVHhPk28mT4R6+dfj95sbQhET9Z5bScf5ejFMHhSd/Ptzm1cj/ALMUJ1K8UwbMzkFQa+F/GaB6jvDZ+T84lkSig2U/W8UMLnoe+GLrxhqVb5ETfvFEoKO0RRTxkY0KHMrfbnDnH91dYpnnHCbhJKuSCs5ecJN/rKXvCrQ5Vww4S/gFrZ/J4Za6fHY4TD7gS6mQCVwc7OWQImprlEB7WI77BFdV14fDeK9KxlG0dfGJQBk4cb51+OI4Cq0E8OVpPpF8XS6a9YxcQJtsdhyLGk9CXeq784Iwgw867Z86GUswKzMdzbAATjFoo3Ez/crWHXvax0vERD9Vw8mWS4dmOdyNPZi33L/dw06f9e8aX7v+zKnzNa0t4POGcMfP2L1V3yuJxIZhNwiwj4XDCoNKvKF6HlOMNJK2gLNbgU4E5w04GqzlNDp5uLCQXZVT5DCv/AQBQJyWr3BiHoesnqvnCKlPnLDWYv8AkYYqd57zh7LV0b7cDqSi1+sSAiuj4TWOnU29jigkR6ABSbZ7BrJYhTX0T6Lrsy/iQU4zdnczQ/hKuBAgnZ8A5JuHWXAG0FpQF8ug5zVBsUdMAWaNh9AzUfLNMjS9ZLDyJdz3nzIfGPvvgKhuH+3A7RAN3AWN2Tr+mZdkPvxbmr6+cApD7nIV7AphmcNCv4z8HHLFhV8uLYXCoW+XGKlW5ttY/BjmbKQ5yhS1InFvPWQXfk7vOUAO7UzlwDPDljn/AN94ycPr8CHf2LDeseoxeWeV0fWaLoyCPzgXZ/JgTt+LeCk/CaY2s5UzpLt5yENXx7Tk4hP73xqy7A685L0BNes52MHKCfxnA7LXvMC19c14k2R7W324KIdK6p/A47gWCEvj1jRNTCEc8xSfGKVc6K1poqLjrp8pw8fpkU+2dWdAqZUKkWY7vjATicxOz7xvOqsBPTXKN9pCcIr8YBgkW9UZ+VxxPR7zVrqVuB3CnqczRyd4vI4gC27+crJ43e3S4AoR0dh1vjWKe+FwnXmdEYb848A96Aoj18ec0c3hu5MeLODKhM8MXn/UT+FfrjqPWIMTl1cR3/cakNRqrtB05YR6iH3Cv1leTbr/AK8u3fp/jBmr/wCvWI3v/wBeMCrENkFANmsP7CWnGOF+Gs2mHD6MUy2K9W/lcHOh2iKNPt4Tbko5jIdK9PI+3EhYYhEaP5wQ9oWfJDuDG+BD8OlGAMi65zY+gFRBlG5Jsc5Ldr7gAOCP8GUsEKpoUqvzh1qFoiKKj841i9cqCGz262ZseCCiF8Na+TLRyJMEDmv2YvBtCo4atLf8s505FYQFIp9M8qh09/ByrXX6wUCh7pdRuusemvUEvYk3o6w0ASwNY7hZ5yzzIHjD574a26/zhDUvPaL+cAvZN8d5ctGEw4gmz1xHmtxDcwX0mpAH7cLCuiHf7we5HHP7N/rOEOFA/eMH9xvIFh5w9v4YFxNmD4M8Iy4HrAU7PC8PMfGdM/E34cHqRuJv4maUzBEIlN8MGub9dxpTUTwCbXX68Z9dw40Y5bJIppzIrkRDkKMqsUmCF+wzWme8NSr/AJYxW12sLJh0vIRgE5FBz4wEndyDHCx4MM6guiW8MANAnQjThlPxg9lD0WiFuFNUKMBt2sB+csqPhbghHP0wEdTQmaieWjMtBc0EsYD2yCkU5PJhgD8c1tFGz5Y9A7AD+L5yso0IbxhuzmdYSRQKZD9LKa/+rhyIXXhdX6/fPIsJOJYawucjnWzJzDxSeebqoe8VyrHbHBd/M+KkEsvZigxaePqzH7mAGfhXfpgA6NAHhHhOhpyHFn9ng4rPeCOejPCMp1jBlb7qLVYJrJojN79CJPG2Vc+Mg5G0Zra6Agb9ZMfxHN0/WOpOE+zAXaRed4hR3Ue9+4XHhdngB7W1Nl7wyaNmoiE2Imn1jaF7EVUqoqrtfeB228i+3MTSd3O3Aef2vYqt5uKcCyWoPxMSBoynP/1PznuSh43/AKfa5948QnsExHXYcnDaY4tTTcEhxinB39YdzQ+BxxrpI6DatZ3RjEEkcuKgcFuQIXV9NRBcXn3Cy0156YPdWDibOLz8yFJzvFZcjFSFnMt0cEaiPuP4zvKObwdsLUsSYoUcuFa4naJ5MzZKrxm6FRtfQWOPxRnGYa4iH3XxiGUTAHtksHJTSJeChdcuM4n2B4S/daRw+GaSQgea5H2bxIp/Z5sN3vPU/qk8YyjcbQ7xLTJd1ypQdnxcMqilSOTJLpGW8YuTl7XjFJrazG6LqCvvC7MQKIOlw4dPDuM+8JxrSxYa3MfOVsidW8W4PwN2V25XWbd3pDDgR0B9ZKEMugQ2nxhklyVC0nW944fJ4GDjGfKS5BsKcZpCC7ClqcmG3SQVa4OZ9mIHKPB87Axms04hYVAFgv4zX6oGBCOZ3m+FpE5feTGxSAW3DuYeAlCy4MMWGsNP2jkpRo+FX8YvYGyCBVeA2fnBcWQknym5rnIW6fPwP4c2rK12iTDutRxGVNuJ+MuU5EV6Wz0d4BjKPF4XzLr4MO4bJEWmseMaLVxAWxu7t+sLDT0skQegfvGLp6pPu+DHmPnNP9lIwK5B8f0bHH6xnrK54NcveIa+gktE+4O2tYWki53h/nZVZuR4DmF+XzTBNahdeMv3qTiSidhWezjBlhRQIVHekp24VkWuDJRbZk1R2jLRAzkxxDfELH1zgM+fvxTgeOR3m6yp554bwgnqkgkS0wjMKq6U/fnPF4h9Yt7tfeJ6LbJE0nwlf4fxm2GBigGN8amLb7tcjpXpM0dun0kz3u4q9vjvjzzIzxcnSdepE9znDu1C1D4HTDFVLRrt8pv1hHMWA/5wvsigUWh40v4yYRq8GwXxyY2Y5gAIG6xw6MejRMK220njqYgpuhFMTVhbxtMbSTppuWydc4hEzQFvhcXBaDZgLyt6xFDum2B63JeLvEVjJB/uytRMn5SyvqM0LpYQ0DwhPEzZsYtADZwjfxnCMA/uBZ/T75tzZzuPpz+XC0Sdl1G2Wpv7ZaoAmmUU8mqmIvh4ctdiCrJVAuQiqX12Zs2LiaSWTUL6xsXXrJJZsQQ4Fesco7GSS+68c0xfMG8DYx554yQODDXT1UcGkx+S6KQGfIbnD5wQW7cJB3tycweVxAe3xjGkrXJgOeWuP9sFJQASejcTlVLg5CQ+QxJ2mzbi4/eYY0WjeLGBWlGtOOlZQJAdPYHeaANq5fAx2TIZaUjm+84ngjFmT/GKfodZd9x41nBu++Kc4DZh5HfYM4MqUwfEZUDRZ0e+9n5z9IjkX+cIOBmOlLyXXnOLlqRCoJM5fzh9uiaAyJBqG+cTYGeC6di4MLEG1HQ5X+RbEMHzkT9vWzVvaY6FgYAH7MVTEK7BVoB50GF6sgrQAh9hxiUAm6mJ30BDxnkFG/j+8Bf6bB39MP8A247moVnhISC6Z2wtHmKd9gNh4LMM4HDdigOPvN8lXi2HY7NY9NwruvWbkDyFvtdZJ0WiAOwNHeK9vt5Ad3TDsPQDV8j5bg924BiafY1xhuppugbhYv3kRKebTQCEPZijU9Ohr8Ha4verW7gYBwkNAYNQkJsxy0flgMwBnnCXmkU8M1uJCPAHgjTppxFd69rJACck4cY8PTj+h+coAHIn7gd4TFkq4rZ4xIXhBgT8uMoVsSSdCXT6w7AJV/G4j5udRvkzfXjDrPCaNc7wIlAVEIHbkuiYkiFcuTTK+E+ZhVzdH9WZNFTmBxbjR+j7HJwLoNOIJMxJd1NxPsGAO47gJ+g5x3xSz4fpmN/40NEa45Xa5cEPCAZWc7U+cSQr3ug+tA9GK8VM8oDBv98+F8UZQju4gFkL2KvSCiJw49arOosOJnx7ZECxGo98UOveUTC+qEXc1uzE62b5n5uGAk8m19jjEiRHeV/RiADhkkCAdExORVpFPffw4KshOdk20fvNMTb6wA49TG/7W/AxPYkw1PVSKwhtdG8BhwN2S+mA96w8+sfvm9aYeLUf/c6AotEiO4d44Vwjs/s6de8j8Eza9uf4y+IXbs8fNj+clapQkd2+/GCAqjWFkAAy2V6IfDpwcDRonkDZurLCRFGlTD14x80cGVy5uNQZu96OfGBpjyK/nEZn/drOrbIFkLtTUwVcsRvpvu4WDHMEBKKOEdITPBvVleDmifeNmBZPobG2r31kTHtcvgcnXXNyXADHlo5gc+3KVXkmoeumi/A+GFX/AIDTkOhas4Ku8M2KojQKAchQ+XJZg0vaDoukj5HBwVEiHy+3PTVPV/oEsdTQjD/LjMi+DAtZ/h4HHCBAPIHGUFBOs76G8EmHMbkRdriv7Y2mSt0hEmLko6TLwOLMcV0ec0JoeoMe4xp58PNV9T0T8FMEnSfgIUPCu5iCGASVqj6wAgdqgO/UZglsnvjGwCfWVjJ8e39OMkVuDYKlx9xbEvHIubRye8eFNN7RacD0kxfj3UrUBB/7cAgoHENaa+yuTLyaI5I0MJrsEFTuaXF50lJc4A19MLanDUfD/pe8EpEUJXlwdP8AwEMAEK0haRpibZxrf4wzs1scHcGkn6B2hQ+cQ9ZVodHxrCP38aw+iy+5lSfnEYtScIXJ2DcFWFgWERHgyThSRNcYn28wnFLYK1X0wPks66SP5fQQOmTesYpxGb7BjphLQflHtQTsaxirX1IagQACQCZK34GkSM54N5+5Nzwkm1jUY1uIJv8AOWvMo/5sEq7vF2mou8Ontjaxq1EbX0wbCq8zBeeXKYNdSfb/AJZcLqb8xgBRO8/sELBubI0eKvBiSbhkE0gN6cPB3OlfqfrCZ7o6+ADPSxf8XOEG1CgfBwM3wfE/4BmNK6xSlfWaSi8hnhnbc1WhN4J4w+2SuFmC2qmPsxoMHpwCzQmhkKA1wYEx25mWbimPDATUVGjwn+cVu/kMpB/AxB3UF49fGUUaOcWkZxpgATu4gPlywSgl+YQNR7zWCF4z2ZDPfqYjct4v6zSgPgw0ntbM0BvtysW/OcB2+Me4+8LYH4mbBViw5f8ABBiYGEMsyMMO4+MN4wCcfjPthPWe04ngwhQmLbGCv+GEoLveCvH9KU/8YwBNTCjX/wDIQl1y3B1ZwAzrMsmHkGsPJK5HrKnjJHQfeT0P6CB4zZr/AIiXDr4yf6ijnoMnPQf0nPQZ6DPQYE/shP6c2cX9z//Z'
            },
        )
        documents = self.env['ir.attachment'].search([
            ('company_id', '=', self.company.id),
            ('res_model', '=', 'account.move'),
            ('description', '=', _('SIE imported file'))
        ])

        document_names = documents.sorted('name').mapped('name')
        file_document_names = tuple(sorted((doc['name'] for doc in file_documents)))
        self.assertSequenceEqual(file_document_names, document_names, 'ir.attachment documents names mismatch file')

        document_content = documents.sorted('datas').mapped('datas')
        file_document_content = tuple(sorted((doc['content'] for doc in file_documents)))
        self.assertSequenceEqual(file_document_content, document_content, 'ir.attachment documents content mismatch file')

    def test_sie_import_consistency(self):
        """ This test just ensures that loading the same file twice will not end up in duplicated data """
        wizard = self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_filedata,
            'include_embedded_documents': True,
        })

        wizard.action_import_sie()
        wizard.action_import_sie()

        # Check accounts and account balances
        accounts = set(self.env['account.account'].search([('company_id', '=', self.company.id)]).mapped('code'))
        self.assertEqual(0, len(self.reference_accounts - accounts))  # All the file accounts have been imported

        endyear_balance = self._get_formatted_balances([
            ('company_id', '=', self.company.id),
            ('date', '<=', '2014-12-31'),
            ('journal_id.code', '=like', 'SIE%'),
        ])

        file_endyear_balance = (
            ("1210", 444050.0), ("1219", -150721.83), ("1240", 175000.00), ("1249", -2916.67), ("1460", 145063.00),
            ("1510", 550231.0), ("1511", 150767.0), ("1515", 16300.0), ("1630", 76563.0), ("1710", 75000.0),
            ("1790", 48007.0), ("1920", 209876.0), ("1940", 604582.78), ("2081", -50000.0), ("2091", -301321.0),
            ("2098", -193179.0), ("2110", -55000.0), ("2129", -55000.0), ("2359", -175000.0), ("2440", -127583.0),
            ("2441", -239232.0), ("2442", -5358.7), ("2510", -261009.0), ("2610", -79255.12), ("2640", 9257.97),
            ("2650", -46626.0), ("2710", -55341.0), ("2731", -70769.4), ("2820", -629.75), ("2890", -70000.0),
            ("2920", -260789.0), ("2941", -80003.0), ("2990", -62000.0), ("3010", -259535.0), ("3020", -230931.0),
            ("3308", -8847.0), ("3740", -2.56), ("3990", -292.5), ("4110", 77284.0), ("4156", 4273.22),
            ("5010", 30000.0), ("5020", 7182.4), ("5050", 1137.6), ("5060", 1627.2), ("5410", 4583.2), ("5460", 303.8),
            ("5500", 5315.2), ("5611", 1506.16), ("5619", 384.0), ("5800", 17386.79), ("5820", 100.0), ("5910", 119.0),
            ("6071", 224.0), ("6110", 37831.2), ("6211", 2021.6), ("6213", 498.0), ("6250", 2991.2), ("6970", 3026.41),
            ("6992", 200.0), ("7010", 85000.0), ("7410", 2500.0), ("7510", 39881.4), ("7690", 950.9), ("7830", 7400.83),
            ("7834", 2916.67),
        )
        self.assertSequenceEqual(file_endyear_balance, endyear_balance, "Balances do not match for the end of the primary year")

        file_startyear_balance = (
            ("1210", 420050.0), ("1219", -143321.0), ("1460", 145063.0), ("1510", 432056.0), ("1515", 16300.0),
            ("1630", 76563.0), ("1710", 75000.0), ("1790", 48007.0), ("1920", 269876.0), ("1940", 387000.0),
            ("2081", -50000.0), ("2091", -301321.0), ("2098", -193179.0), ("2110", -55000.0), ("2129", -55000.0),
            ("2440", -240632.0), ("2510", -261009.0), ("2650", -36897.0), ("2710", -29876.0), ("2731", -30888.0),
            ("2890", -70000.0), ("2920", -260789.0), ("2941", -80003.0), ("2990", -62000.0),
        )
        startyear_balance = self._get_formatted_balances([
            ('company_id', '=', self.company.id),
            ('date', '<=', '2014-01-01'),
            ('journal_id.code', '=like', 'SIE%'),
        ])
        self.assertSequenceEqual(file_startyear_balance, startyear_balance, "Balances do not match for the start of the primary year")

        # Check journals
        journals = self.env['account.journal'].search([('code', '=', 'SIE')])
        self.assertTrue(journals.ids, "Journal creation failed")

        # Check partners
        file_ids = (
            '1002', '1004', '1006', '1007', '1008', '1009', '1010', '1011', '1012', '1013', '1015', '1017', '1019',
            '4545-6633', '7410-2365', '7788-3333', '7841-9999', '7856-8999', '7888-9999', '7899-5558', '7901', '7902',
            '7903', '8798-8989', '8977-9665', '8978-5621', '8978-6235', '8978-8999'
        )
        file_partners = [_('SIE undefined imported partner - %s', name) for name in file_ids]
        file_partners += ['Åkesson & co', 'Decoration A/S', 'Reklamförbundet org', 'Simone & Partners - Sprl']
        file_bad_partners = [
            'SIE undefined imported partner - 1000',
            'SIE undefined imported partner - DK122333',
            'SIE undefined imported partner - 1111-2356',
            'SIE undefined imported partner - 1022',
        ]
        # These should not be created, they correspond to the file_ids of the named partners

        n_partners = self.env['res.partner'].search_count([('name', 'in', file_partners + file_bad_partners)])
        self.assertEqual(len(file_partners), n_partners, "Partners created on import don't match the file")

        # Check documents
        file_documents = (  # Only those should be imported as the moves for the others are on corrected moves
            {
                'name': 'SIE_1 - eSKD moms 1401 - 140203 133435.xml',
                'content': b'PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iSVNPLTg4NTktMSI/Pg0KPCFET0NUWVBFIGVTS0RVcGxvYWQgUFVCTElDICItLy9Ta2F0dGV2ZXJrZXQsIFN3ZWRlbi8vRFREIFNrYXR0ZXZlcmtldCBlU0tEVXBsb2FkLURURCBWZXJzaW9uIDMuMC8vU1YiICJodHRwczovL3d3dzEuc2thdHRldmVya2V0LnNlL2RlbW9lc2tkL2VTS0RVcGxvYWRfM3AwLmR0ZCI+DQo8ZVNLRFVwbG9hZCBWZXJzaW9uPSIzLjAiPg0KCTxPcmdOcj41NTU1NTUtNTU1NTwvT3JnTnI+DQoJPE1vbXM+DQoJCTxQZXJpb2Q+MjAxNDAxPC9QZXJpb2Q+DQoJCTxGb3JzTW9tc0VqQW5uYW4+MTczNzM4PC9Gb3JzTW9tc0VqQW5uYW4+PFV0dGFnTW9tcz4wPC9VdHRhZ01vbXM+PFVsYWdNYXJnYmVzaz4wPC9VbGFnTWFyZ2Jlc2s+PEh5cmlua29tc3RGcml2PjA8L0h5cmlua29tc3RGcml2PjxJbmtvcFZhcnVBbm5hdEVnPjQyNzM8L0lua29wVmFydUFubmF0RWc+PElua29wVGphbnN0QW5uYXRFZz4wPC9JbmtvcFRqYW5zdEFubmF0RWc+PElua29wVGphbnN0VXRvbUVnPjA8L0lua29wVGphbnN0VXRvbUVnPjxJbmtvcFZhcnVTdmVyaWdlPjA8L0lua29wVmFydVN2ZXJpZ2U+PElua29wVGphbnN0U3ZlcmlnZT4wPC9JbmtvcFRqYW5zdFN2ZXJpZ2U+PEZvcnNWYXJ1QW5uYXRFZz4wPC9Gb3JzVmFydUFubmF0RWc+PEZvcnNWYXJ1VXRvbUVnPjA8L0ZvcnNWYXJ1VXRvbUVnPjxJbmtvcFZhcnVNZWxsYW4zcD4wPC9JbmtvcFZhcnVNZWxsYW4zcD48Rm9yc1ZhcnVNZWxsYW4zcD4wPC9Gb3JzVmFydU1lbGxhbjNwPjxGb3JzVGpTa3NrQW5uYXRFZz44ODQ3PC9Gb3JzVGpTa3NrQW5uYXRFZz48Rm9yc1RqT3ZyVXRvbUVnPjA8L0ZvcnNUak92clV0b21FZz48Rm9yc0tvcGFyZVNrc2tTdmVyaWdlPjA8L0ZvcnNLb3BhcmVTa3NrU3ZlcmlnZT48Rm9yc092cmlndD4wPC9Gb3JzT3ZyaWd0PjxNb21zVXRnSG9nPjQzNDM0PC9Nb21zVXRnSG9nPjxNb21zVXRnTWVkZWw+MDwvTW9tc1V0Z01lZGVsPjxNb21zVXRnTGFnPjA8L01vbXNVdGdMYWc+PE1vbXNJbmtvcFV0Z0hvZz4xMDY4PC9Nb21zSW5rb3BVdGdIb2c+PE1vbXNJbmtvcFV0Z01lZGVsPjA8L01vbXNJbmtvcFV0Z01lZGVsPjxNb21zSW5rb3BVdGdMYWc+MDwvTW9tc0lua29wVXRnTGFnPjxNb21zSW5nQXZkcj4zNDc3MzwvTW9tc0luZ0F2ZHI+PE1vbXNCZXRhbGE+OTcyOTwvTW9tc0JldGFsYT48L01vbXM+DQo8L2VTS0RVcGxvYWQ+DQo='
            },
            {
                'name': 'SIE_Betalfil Bg - 1 - 140102 112969.bgc',
                'content': b'MTEwMDA3MzQ2MjMyMTQwMTAyTEVWRVJBTlTWUlNCRVRBTE5JTkdBUiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICANCjEzRXJ0IGZha3R1cmFuciAgIFblciByZWYgICAgICAgICBOZXR0byAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQoxNDAwMDAwMDAwMDAyMyAgICAgICAgICAgICAgMSAgICAgICAgMDAwMDAwMDk3ODAwMTQwMTIwICAgICAxOjEgICAgICAgICAgICAgICAgIA0KMTQwMDAwMDAwMDAwNzg5NSAgICAgICAgICAgIDExICAgICAgIDAwMDAwMTA0NjAwMDE0MDEyOCAgICAgMToyICAgICAgICAgICAgICAgICANCjE0MDAwMDAwMDAwMDU3MCAgICAgICAgICAgICAxMiAgICAgICAwMDAwMDAwNzA0MDAxNDAxMjggICAgIDE6MyAgICAgICAgICAgICAgICAgDQoyOTAwMDczNDYyMzIwMDAwMDAwMzAwMDAwMTIxNDIwMCAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIA0K'
            },
            {
                'name': 'SIE_Betalfil Bg - 1 - 140102 113283.bgc',
                'content': b'MTEwMDA3MzQ2MjMyMTQwMTAyTEVWRVJBTlTWUlNCRVRBTE5JTkdBUiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICANCjEzRXJ0IGZha3R1cmFuciAgIFblciByZWYgICAgICAgICBOZXR0byAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQoxNDAwMDAwMDAwMDAyMyAgICAgICAgICAgICAgMSAgICAgICAgMDAwMDAwMDQ4OTAwMTQwMTIwICAgICAyOjEgICAgICAgICAgICAgICAgIA0KMTQwMDAwMDAwMDAwNzg5NSAgICAgICAgICAgIDExICAgICAgIDAwMDAwMDUyMzAwMDE0MDEyOCAgICAgMjoyICAgICAgICAgICAgICAgICANCjI5MDAwNzM0NjIzMjAwMDAwMDAyMDAwMDAwNTcxOTAwICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgDQo='
            },
            {
                'name': 'SIE_EAGWP.jpg',
                'content': b'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'
            },
        )
        documents = self.env['ir.attachment'].search([
            ('company_id', '=', self.company.id),
            ('res_model', '=', 'account.move'),
            ('description', '=', _('SIE imported file'))
        ])

        document_names = documents.sorted('name').mapped('name')
        file_document_names = tuple(sorted((doc['name'] for doc in file_documents)))
        self.assertSequenceEqual(file_document_names, document_names, 'ir.attachment documents names mismatch file')

        document_content = documents.sorted('datas').mapped('datas')
        file_document_content = tuple(sorted((doc['content'] for doc in file_documents)))
        self.assertSequenceEqual(file_document_content, document_content, 'ir.attachment documents content mismatch file')

    def test_sie_combination(self):
        """ This test just ensures that loading different files will not end up in duplicated data """
        sie_wizard = self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_filedata,
            'include_embedded_documents': True,
        })
        sie_entry_wizard = self.env['l10n_se.sie.import.wizard'].create({
            'attachment_id': self.sie_entry_filedata,
            'include_embedded_documents': True,
        })
        sie_wizard.action_import_sie()
        sie_entry_wizard.action_import_sie()


        # Check accounts and balances
        accounts = set(self.env['account.account'].search([('company_id', '=', self.company.id)]).mapped('code'))
        self.assertEqual(0, len(self.reference_accounts - accounts))
        balance = self._get_formatted_balances([('company_id', '=', self.company.id), ('journal_id.code', '=like', 'SIE%')])

        file_balance = (
            ("1210", 444050.0), ("1219", -150721.83), ("1240", 175000.00), ("1249", -2916.67), ("1460", 145063.00),
            ("1510", 550231.0), ("1511", 150767.0), ("1515", 16300.0), ("1630", 76563.0), ("1710", 75000.0),
            ("1790", 48007.0), ("1920", 209876.0), ("1940", 604567.03), ("2081", -50000.0), ("2091", -301321.0),
            ("2098", -193179.0), ("2110", -55000.0), ("2129", -55000.0), ("2359", -175000.0), ("2440", -127583.0),
            ("2441", -239232.0), ("2442", -5358.7), ("2510", -261009.0), ("2610", -79255.12), ("2640", 9257.97),
            ("2650", -46626.0), ("2710", -55341.0), ("2731", -70769.4), ("2820", -629.75), ("2890", -70000.0),
            ("2920", -260789.0), ("2941", -80003.0), ("2990", -62000.0), ("3010", -259535.0), ("3020", -230931.0),
            ("3308", -8847.0), ("3740", -2.56), ("3990", -292.5), ("4110", 77284.0), ("4156", 4273.22),
            ("5010", 30000.0), ("5020", 7182.4), ("5050", 1137.6), ("5060", 1627.2), ("5410", 4583.2), ("5460", 303.8),
            ("5500", 5315.2), ("5611", 1506.16), ("5619", 384.0), ("5800", 17386.79), ("5820", 100.0), ("5910", 119.0),
            ("6071", 239.75), ("6110", 37831.2), ("6211", 2021.6), ("6213", 498.0), ("6250", 2991.2), ("6970", 3026.41),
            ("6992", 200.0), ("7010", 85000.0), ("7410", 2500.0), ("7510", 39881.4), ("7690", 950.9), ("7830", 7400.83),
            ("7834", 2916.67),
        )  # The changing amounts between the simple SIE and the SIE + SIE entry are in accounts 1940 & 6071
        self.assertSequenceEqual(file_balance, balance, "Balances do not match for the end of the year")

        # Check journals
        journals = self.env['account.journal'].search([('code', '=', 'SIE')])
        self.assertTrue(journals.ids, "Journal creation failed")

        # Check partners
        file_ids = (
            '1002', '1004', '1006', '1007', '1008', '1009', '1010', '1011', '1012', '1013', '1015', '1017', '1019',
            '4545-6633', '7410-2365', '7788-3333', '7841-9999', '7856-8999', '7888-9999', '7899-5558', '7901', '7902',
            '7903', '8798-8989', '8977-9665', '8978-5621', '8978-6235', '8978-8999'
        )
        file_partners = [_('SIE undefined imported partner - %s', name) for name in file_ids]
        file_partners += [
            'Åkesson & co', 'Decoration A/S', 'Reklamförbundet org', 'Simone & Partners - Sprl', 'Svenska Statoil AB', 'Hyresvärden AB'
        ]
        file_bad_partners = [
            'SIE undefined imported partner - 1000',
            'SIE undefined imported partner - DK122333',
            'SIE undefined imported partner - 1111-2356',
            'SIE undefined imported partner - 1022',
            'SIE undefined imported partner - 7',
            'SIE undefined imported partner - 14',
        ]
        # These should not be created, they correspond to the file_ids of the named partners

        n_partners = self.env['res.partner'].search_count([('name', 'in', file_partners + file_bad_partners)])
        self.assertEqual(len(file_partners), n_partners, "Partners created on import don't match the file")

        # Check documents
        n_attachments = self.env['ir.attachment'].search_count([
            ('company_id', '=', self.company.id),
            ('res_model', '=', 'account.move'),
            ('description', '=', _('SIE imported file')),
        ])
        self.assertEqual(4, n_attachments, 'ir.attachment documents imported mismatch files')
